/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.cram.structure;

import htsjdk.samtools.cram.common.CramVersions;
import htsjdk.samtools.cram.io.CRC32InputStream;
import htsjdk.samtools.cram.io.CRC32OutputStream;
import htsjdk.samtools.cram.io.CramInt;
import htsjdk.samtools.cram.io.ExternalCompression;
import htsjdk.samtools.cram.io.ITF8;
import htsjdk.samtools.cram.io.InputStreamUtils;
import htsjdk.samtools.cram.structure.BlockCompressionMethod;
import htsjdk.samtools.cram.structure.BlockContentType;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;

public class Block {
    private BlockCompressionMethod method;
    private BlockContentType contentType;
    private int contentId;
    private int compressedContentSize;
    private int rawContentSize;
    private byte[] rawContent;
    private byte[] compressedContent;

    public Block() {
    }

    public static Block readFromInputStream(int major, InputStream inputStream) throws IOException {
        boolean v3OrHigher;
        Block block = new Block();
        boolean bl = v3OrHigher = major >= CramVersions.CRAM_v3.major;
        if (v3OrHigher) {
            inputStream = new CRC32InputStream(inputStream);
        }
        block.setMethod(BlockCompressionMethod.values()[inputStream.read()]);
        int contentTypeId = inputStream.read();
        block.setContentType(BlockContentType.values()[contentTypeId]);
        block.setContentId(ITF8.readUnsignedITF8(inputStream));
        block.compressedContentSize = ITF8.readUnsignedITF8(inputStream);
        block.rawContentSize = ITF8.readUnsignedITF8(inputStream);
        block.compressedContent = new byte[block.compressedContentSize];
        InputStreamUtils.readFully(inputStream, block.compressedContent, 0, block.compressedContent.length);
        if (v3OrHigher) {
            int actualChecksum = ((CRC32InputStream)inputStream).getCRC32();
            int checksum = CramInt.int32((InputStream)inputStream);
            if (checksum != actualChecksum) {
                throw new RuntimeException(String.format("Block CRC32 mismatch: %04x vs %04x", checksum, actualChecksum));
            }
        }
        block.uncompress();
        return block;
    }

    public static Block buildNewSliceHeaderBlock(byte[] rawContent) {
        return new Block(BlockContentType.MAPPED_SLICE, rawContent);
    }

    public static Block buildNewCore(byte[] rawContent) {
        return new Block(BlockContentType.CORE, rawContent);
    }

    public static Block buildNewFileHeaderBlock(byte[] rawContent) {
        Block block = new Block(BlockContentType.FILE_HEADER, rawContent);
        block.compress();
        return block;
    }

    private Block(BlockContentType contentType, byte[] rawContent) {
        this.setMethod(BlockCompressionMethod.RAW);
        this.setContentType(contentType);
        this.setContentId(0);
        if (rawContent != null) {
            this.setRawContent(rawContent);
        }
    }

    public String toString() {
        String raw = this.rawContent == null ? "NULL" : Arrays.toString(Arrays.copyOf(this.rawContent, Math.min(5, this.rawContent.length)));
        String comp = this.compressedContent == null ? "NULL" : Arrays.toString(Arrays.copyOf(this.compressedContent, Math.min(5, this.compressedContent.length)));
        return String.format("method=%s, type=%s, id=%d, raw size=%d, compressed size=%d, raw=%s, comp=%s.", this.getMethod().name(), this.getContentType().name(), this.getContentId(), this.rawContentSize, this.compressedContentSize, raw, comp);
    }

    boolean isCompressed() {
        return this.compressedContent != null;
    }

    boolean isUncompressed() {
        return this.rawContent != null;
    }

    public void setRawContent(byte[] raw) {
        this.rawContent = raw;
        this.rawContentSize = raw == null ? 0 : raw.length;
        this.compressedContent = null;
        this.compressedContentSize = 0;
    }

    public byte[] getRawContent() {
        if (this.rawContent == null) {
            this.uncompress();
        }
        return this.rawContent;
    }

    public int getRawContentSize() {
        return this.rawContentSize;
    }

    public void setContent(byte[] raw, byte[] compressed) {
        this.rawContent = raw;
        this.compressedContent = compressed;
        this.rawContentSize = raw == null ? 0 : raw.length;
        this.compressedContentSize = compressed == null ? 0 : compressed.length;
    }

    void setCompressedContent(byte[] compressed) {
        this.compressedContent = compressed;
        this.compressedContentSize = compressed == null ? 0 : compressed.length;
        this.rawContent = null;
        this.rawContentSize = 0;
    }

    byte[] getCompressedContent() {
        if (this.compressedContent == null) {
            this.compress();
        }
        return this.compressedContent;
    }

    private void compress() {
        if (this.compressedContent != null || this.rawContent == null) {
            return;
        }
        switch (this.getMethod()) {
            case RAW: {
                this.compressedContent = this.rawContent;
                this.compressedContentSize = this.rawContentSize;
                break;
            }
            case GZIP: {
                try {
                    this.compressedContent = ExternalCompression.gzip(this.rawContent);
                }
                catch (IOException e) {
                    throw new RuntimeException("This should have never happened.", e);
                }
                this.compressedContentSize = this.compressedContent.length;
                break;
            }
            case RANS: {
                this.compressedContent = ExternalCompression.rans(this.rawContent, 1);
                this.compressedContentSize = this.compressedContent.length;
                break;
            }
        }
    }

    private void uncompress() {
        if (this.rawContent != null || this.compressedContent == null) {
            return;
        }
        switch (this.getMethod()) {
            case RAW: {
                this.rawContent = this.compressedContent;
                this.rawContentSize = this.compressedContentSize;
                break;
            }
            case GZIP: {
                try {
                    this.rawContent = ExternalCompression.gunzip(this.compressedContent);
                    break;
                }
                catch (IOException e) {
                    throw new RuntimeException("This should have never happened.", e);
                }
            }
            case BZIP2: {
                try {
                    this.rawContent = ExternalCompression.unbzip2(this.compressedContent);
                    break;
                }
                catch (IOException e) {
                    throw new RuntimeException("This should have never happened.", e);
                }
            }
            case LZMA: {
                try {
                    this.rawContent = ExternalCompression.unxz(this.compressedContent);
                    break;
                }
                catch (IOException e) {
                    throw new RuntimeException("This should have never happened.", e);
                }
            }
            case RANS: {
                this.rawContent = ExternalCompression.unrans(this.compressedContent);
                break;
            }
            default: {
                throw new RuntimeException("Unknown block compression method: " + this.getMethod().name());
            }
        }
    }

    public void write(int major, OutputStream outputStream) throws IOException {
        if (major >= CramVersions.CRAM_v3.major) {
            CRC32OutputStream crc32OutputStream = new CRC32OutputStream(outputStream);
            this.doWrite(crc32OutputStream);
            outputStream.write(crc32OutputStream.getCrc32_LittleEndian());
        } else {
            this.doWrite(outputStream);
        }
    }

    private void doWrite(OutputStream outputStream) throws IOException {
        if (!this.isCompressed()) {
            this.compress();
        }
        if (!this.isUncompressed()) {
            this.uncompress();
        }
        outputStream.write(this.getMethod().ordinal());
        outputStream.write(this.getContentType().ordinal());
        ITF8.writeUnsignedITF8(this.getContentId(), outputStream);
        ITF8.writeUnsignedITF8(this.compressedContentSize, outputStream);
        ITF8.writeUnsignedITF8(this.rawContentSize, outputStream);
        outputStream.write(this.getCompressedContent());
    }

    BlockCompressionMethod getMethod() {
        return this.method;
    }

    public void setMethod(BlockCompressionMethod method) {
        this.method = method;
    }

    public BlockContentType getContentType() {
        return this.contentType;
    }

    public void setContentType(BlockContentType contentType) {
        this.contentType = contentType;
    }

    public int getContentId() {
        return this.contentId;
    }

    public void setContentId(int contentId) {
        this.contentId = contentId;
    }

    public int getCompressedContentSize() {
        return this.compressedContentSize;
    }
}

