/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.util;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMUtils;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.fastq.FastqReader;
import htsjdk.samtools.fastq.FastqRecord;
import htsjdk.samtools.util.CloseableIterator;
import htsjdk.samtools.util.FastqQualityFormat;
import htsjdk.samtools.util.Log;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.Set;

public class QualityEncodingDetector {
    private QualityRecordAggregator qualityAggregator = new QualityRecordAggregator();
    public static final long DEFAULT_MAX_RECORDS_TO_ITERATE = 10000L;
    private static final Log log = Log.getInstance(QualityEncodingDetector.class);

    public long add(long maxRecords, FastqReader ... readers) {
        Iterator<FastqRecord> iterator2 = QualityEncodingDetector.generateInterleavedFastqIterator(readers);
        long recordCount = 0L;
        while (iterator2.hasNext() && recordCount++ != maxRecords) {
            this.add(iterator2.next());
        }
        log.debug(String.format("Read %s records from %s.", recordCount, Arrays.toString(readers)));
        return recordCount;
    }

    public long add(long maxRecords, SamReader reader) {
        return this.add(maxRecords, reader.iterator());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long add(long maxRecords, CloseableIterator<SAMRecord> iterator2, boolean useOriginalQualities) {
        long recordCount = 0L;
        try {
            while (iterator2.hasNext() && recordCount++ != maxRecords) {
                this.add((SAMRecord)iterator2.next(), useOriginalQualities);
            }
            long l = recordCount;
            return l;
        }
        finally {
            iterator2.close();
        }
    }

    public long add(long maxRecords, CloseableIterator<SAMRecord> iterator2) {
        return this.add(maxRecords, iterator2, false);
    }

    public void add(FastqRecord fastqRecord) {
        this.qualityAggregator.add(fastqRecord);
    }

    public void add(SAMRecord samRecord, boolean useOriginalQualities) {
        this.qualityAggregator.add(samRecord, useOriginalQualities);
    }

    public void add(SAMRecord samRecord) {
        this.add(samRecord, false);
    }

    public boolean isDeterminationAmbiguous() {
        return this.generateCandidateQualities(true).size() > 1;
    }

    public EnumSet<FastqQualityFormat> generateCandidateQualities(boolean checkExpected) {
        EnumSet<FastqQualityFormat> candidateFormats = EnumSet.allOf(FastqQualityFormat.class);
        Set<Integer> observedAsciiQualities = this.qualityAggregator.getObservedAsciiQualities();
        if (observedAsciiQualities.isEmpty()) {
            throw new SAMException("Cannot determine candidate qualities: no qualities found.");
        }
        for (QualityScheme scheme : QualityScheme.values()) {
            Iterator<Integer> qualityBinIterator = observedAsciiQualities.iterator();
            ArrayList<Range> remainingExpectedValueRanges = new ArrayList<Range>(scheme.expectedAsciiRanges);
            while (qualityBinIterator.hasNext()) {
                int quality = qualityBinIterator.next();
                if (!scheme.asciiRange.contains(quality)) {
                    candidateFormats.remove((Object)scheme.qualityFormat);
                }
                Iterator expectedValueRangeIterator = remainingExpectedValueRanges.iterator();
                while (expectedValueRangeIterator.hasNext()) {
                    if (!((Range)expectedValueRangeIterator.next()).contains(quality)) continue;
                    expectedValueRangeIterator.remove();
                }
            }
            if (remainingExpectedValueRanges.isEmpty() || !checkExpected) continue;
            candidateFormats.remove((Object)scheme.qualityFormat);
        }
        return candidateFormats;
    }

    private static Iterator<FastqRecord> generateInterleavedFastqIterator(final FastqReader ... readers) {
        return new Iterator<FastqRecord>(){
            private Queue<Iterator<FastqRecord>> queue = new LinkedList<Iterator<FastqRecord>>();
            {
                for (FastqReader reader : readers) {
                    this.queue.add(reader.iterator());
                }
            }

            @Override
            public boolean hasNext() {
                while (!this.queue.isEmpty()) {
                    if (this.queue.peek().hasNext()) {
                        return true;
                    }
                    this.queue.poll();
                }
                return false;
            }

            @Override
            public FastqRecord next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                Iterator<FastqRecord> i = this.queue.poll();
                FastqRecord result = i.next();
                this.queue.offer(i);
                return result;
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    public static FastqQualityFormat detect(long maxRecords, FastqReader ... readers) {
        QualityEncodingDetector detector = new QualityEncodingDetector();
        long recordCount = detector.add(maxRecords, readers);
        log.debug(String.format("Read %s records from %s.", recordCount, Arrays.toString(readers)));
        return detector.generateBestGuess(FileContext.FASTQ, null);
    }

    public static FastqQualityFormat detect(FastqReader ... readers) {
        return QualityEncodingDetector.detect(10000L, readers);
    }

    public static FastqQualityFormat detect(long maxRecords, CloseableIterator<SAMRecord> iterator2, boolean useOriginalQualities) {
        QualityEncodingDetector detector = new QualityEncodingDetector();
        long recordCount = detector.add(maxRecords, iterator2, useOriginalQualities);
        log.debug(String.format("Read %s records.", recordCount));
        return detector.generateBestGuess(FileContext.SAM, null);
    }

    public static FastqQualityFormat detect(long maxRecords, CloseableIterator<SAMRecord> iterator2) {
        return QualityEncodingDetector.detect(maxRecords, iterator2, false);
    }

    public static FastqQualityFormat detect(long maxRecords, SamReader reader) {
        return QualityEncodingDetector.detect(maxRecords, reader.iterator());
    }

    public static FastqQualityFormat detect(SamReader reader) {
        return QualityEncodingDetector.detect(10000L, reader);
    }

    public static FastqQualityFormat detect(SamReader reader, FastqQualityFormat expectedQualityFormat) {
        QualityEncodingDetector detector = new QualityEncodingDetector();
        long recordCount = detector.add(10000L, reader.iterator());
        log.debug(String.format("Read %s records from %s.", recordCount, reader));
        return detector.generateBestGuess(FileContext.SAM, expectedQualityFormat);
    }

    public FastqQualityFormat generateBestGuess(FileContext context, FastqQualityFormat expectedQuality) {
        if (null != expectedQuality) {
            EnumSet<FastqQualityFormat> possibleFormats = this.generateCandidateQualities(false);
            if (possibleFormats.contains((Object)expectedQuality)) {
                return expectedQuality;
            }
            throw new SAMException(String.format("The quality values do not fall in the range appropriate for the expected quality of %s.", expectedQuality.name()));
        }
        EnumSet<FastqQualityFormat> possibleFormats = this.generateCandidateQualities(true);
        switch (possibleFormats.size()) {
            case 1: {
                return (FastqQualityFormat)((Object)possibleFormats.iterator().next());
            }
            case 2: {
                if (possibleFormats.equals(EnumSet.of(FastqQualityFormat.Illumina, FastqQualityFormat.Solexa))) {
                    return FastqQualityFormat.Illumina;
                }
                if (possibleFormats.equals(EnumSet.of(FastqQualityFormat.Illumina, FastqQualityFormat.Standard))) {
                    switch (context) {
                        case FASTQ: {
                            return FastqQualityFormat.Illumina;
                        }
                        case SAM: {
                            return FastqQualityFormat.Standard;
                        }
                    }
                } else {
                    if (possibleFormats.equals(EnumSet.of(FastqQualityFormat.Standard, FastqQualityFormat.Solexa))) {
                        return FastqQualityFormat.Standard;
                    }
                    throw new SAMException("Unreachable code.");
                }
            }
            case 3: {
                throw new SAMException("The quality format cannot be determined: no formats were excluded.");
            }
            case 0: {
                throw new SAMException("The quality format cannot be determined: all formats were excluded.");
            }
        }
        throw new SAMException("Unreachable code.");
    }

    private static class QualityRecordAggregator {
        private Set<Integer> observedAsciiQualities = new HashSet<Integer>();

        private QualityRecordAggregator() {
        }

        public Set<Integer> getObservedAsciiQualities() {
            return Collections.unmodifiableSet(this.observedAsciiQualities);
        }

        public void add(FastqRecord fastqRecord) {
            this.addAsciiQuality(fastqRecord.getBaseQualityString().getBytes());
        }

        public void add(SAMRecord samRecord, boolean useOriginalQualities) {
            this.addAsciiQuality(useOriginalQualities && samRecord.getOriginalBaseQualities() != null ? SAMUtils.phredToFastq(samRecord.getOriginalBaseQualities()).getBytes() : samRecord.getBaseQualityString().getBytes());
        }

        public void add(SAMRecord samRecord) {
            this.add(samRecord, false);
        }

        private void addAsciiQuality(byte ... asciiQualities) {
            for (byte asciiQuality : asciiQualities) {
                this.observedAsciiQualities.add(Integer.valueOf(asciiQuality));
            }
        }
    }

    static enum QualityScheme {
        Phred(new Range(0, 93), new Range(33, 126), Arrays.asList(new Range(33, 58)), FastqQualityFormat.Standard),
        Solexa(new Range(-5, 62), new Range(59, 126), new ArrayList<Range>(), FastqQualityFormat.Solexa),
        Illumina(new Range(0, 62), new Range(64, 126), new ArrayList<Range>(), FastqQualityFormat.Illumina);

        final Range rawRange;
        final Range asciiRange;
        final List<Range> expectedAsciiRanges;
        final FastqQualityFormat qualityFormat;

        private QualityScheme(Range rawRange, Range asciiRange, List<Range> expectedAsciiRanges, FastqQualityFormat qualityFormat) {
            this.rawRange = rawRange;
            this.asciiRange = asciiRange;
            this.expectedAsciiRanges = expectedAsciiRanges;
            this.qualityFormat = qualityFormat;
        }
    }

    static class Range {
        final int low;
        final int high;

        Range(int low, int high) {
            this.low = low;
            this.high = high;
        }

        boolean contains(int value) {
            return value <= this.high && value >= this.low;
        }
    }

    public static enum FileContext {
        FASTQ,
        SAM;

    }
}

