/* ----------------------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   http://lammps.sandia.gov, Sandia National Laboratories
   Steve Plimpton, sjplimp@sandia.gov

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#include "atom.h"
#include "atom_vec_body.h"
#include "atom_vec_ellipsoid.h"
#include "atom_vec_hybrid.h"
#include "atom_vec_line.h"
#include "atom_vec_tri.h"
#include "body.h"
#include "input.h"
#include "lammps.h"
#include "math_const.h"
#include "utils.h"
#include "gmock/gmock.h"
#include "gtest/gtest.h"

#include <cmath>
#include <cstdio>
#include <cstring>
#include <mpi.h>

#if !defined(_FORTIFY_SOURCE) || (_FORTIFY_SOURCE == 0)
#if defined(__INTEL_COMPILER)
#define _do_nothing
#elif defined(__clang__)
#pragma clang optimize off
#elif defined(__GNUC__)
#if (__GNUC__ > 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ >= 9))
#pragma GCC optimize("no-var-tracking-assignments", "O0")
#else
#pragma GCC optimize("no-var-tracking-assignments")
#endif
#else
#define _do_nothing
#endif
#endif

#define GETIDX(i) lmp->atom->map(i)

using LAMMPS_NS::utils::split_words;

// whether to print verbose output (i.e. not capturing LAMMPS screen output).
bool verbose = false;

namespace LAMMPS_NS {
using ::testing::Eq;

class AtomStyleTest : public ::testing::Test {
protected:
    LAMMPS *lmp;

    void SetUp() override
    {
        const char *args[] = {"SimpleCommandsTest", "-log", "none", "-echo", "screen", "-nocite"};
        char **argv        = (char **)args;
        int argc           = sizeof(args) / sizeof(char *);
        if (!verbose) ::testing::internal::CaptureStdout();
        lmp = new LAMMPS(argc, argv, MPI_COMM_WORLD);
        if (!verbose) ::testing::internal::GetCapturedStdout();
        ASSERT_NE(lmp, nullptr);
        if (!verbose) ::testing::internal::CaptureStdout();
        lmp->input->one("units real");
        lmp->input->one("dimension 3");
        lmp->input->one("pair_style zero 4.0");
        lmp->input->one("region box block -4 4 -4 4 -4 4");
        if (!verbose) ::testing::internal::GetCapturedStdout();
    }

    void TearDown() override
    {
        if (!verbose) ::testing::internal::CaptureStdout();
        delete lmp;
        if (!verbose) ::testing::internal::GetCapturedStdout();
        remove("test_atom_styles.data");
        remove("input_atom_styles.data");
        remove("test_atom_styles.restart");
    }
};

TEST_F(AtomStyleTest, atomic)
{
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 0);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 0);
    ASSERT_EQ(lmp->atom->radius_flag, 0);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 0);
    ASSERT_EQ(lmp->atom->angmom_flag, 0);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_EQ(lmp->atom->angmom, nullptr);
    ASSERT_EQ(lmp->atom->torque, nullptr);
    ASSERT_EQ(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_EQ(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style charge");
    lmp->input->one("atom_style atomic");
    if (!verbose) ::testing::internal::GetCapturedStdout();

    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);

    ASSERT_EQ(lmp->atom->molecule_flag, 0);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);

    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->q, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_modify map hash");
    lmp->input->one("create_box 2 box");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("mass 1 4.0");
    lmp->input->one("mass 2 2.4");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 2);
    ASSERT_EQ(lmp->atom->map_user, 2);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("atom_style atomic");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("atom_modify map array");
    lmp->input->one("units real");
    lmp->input->one("read_data test_atom_styles.data");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    auto x   = lmp->atom->x;
    auto v   = lmp->atom->v;
    auto tag = lmp->atom->tag;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);

    ASSERT_NEAR(lmp->atom->mass[1], 4.0, 1.0e-14);
    ASSERT_NEAR(lmp->atom->mass[2], 2.4, 1.0e-14);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 2);
    ASSERT_EQ(lmp->atom->nlocal, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);

    x = lmp->atom->x;
    v = lmp->atom->v;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);

    ASSERT_NEAR(lmp->atom->mass[1], 4.0, 1.0e-14);
    ASSERT_NEAR(lmp->atom->mass[2], 2.4, 1.0e-14);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 3);
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->map_tag_max, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("comm_style tiled");
    lmp->input->one("change_box all triclinic");
    lmp->input->one("replicate 2 2 2");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->map_tag_max, 16);
    x   = lmp->atom->x;
    tag = lmp->atom->tag;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 6.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(7)][0], 6.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(7)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(7)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(8)][0], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(8)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(8)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(9)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(9)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(9)][2], 8.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(10)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(10)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(10)][2], 7.9, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(11)][0], 6.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(11)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(11)][2], 8.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(12)][0], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(12)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(12)][2], 7.9, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(13)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(13)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(13)][2], 8.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(14)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(14)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(14)][2], 7.9, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(15)][0], 6.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(15)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(15)][2], 8.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(16)][0], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(16)][1], 10.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(16)][2], 7.9, 1.0e-14);
}

TEST_F(AtomStyleTest, charge)
{
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style charge");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 0);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 1);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 0);
    ASSERT_EQ(lmp->atom->radius_flag, 0);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 0);
    ASSERT_EQ(lmp->atom->angmom_flag, 0);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_NE(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_EQ(lmp->atom->angmom, nullptr);
    ASSERT_EQ(lmp->atom->torque, nullptr);
    ASSERT_EQ(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_EQ(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 2 box");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("mass 1 4.0");
    lmp->input->one("mass 2 2.4");
    lmp->input->one("set atom 1 charge -0.5");
    lmp->input->one("set atom 2 charge  0.5");
    lmp->input->one("set atom 3 charge -1.0");
    lmp->input->one("set atom 4 charge  1.0");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("atom_style charge");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("units real");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->q_flag, 1);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    auto x = lmp->atom->x;
    auto v = lmp->atom->v;
    auto q = lmp->atom->q;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(1)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(2)], 0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(3)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(4)], 1.0, 1.0e-14);

    ASSERT_NEAR(lmp->atom->mass[1], 4.0, 1.0e-14);
    ASSERT_NEAR(lmp->atom->mass[2], 2.4, 1.0e-14);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    EXPECT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    lmp->input->one("read_restart test_atom_styles.restart");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("charge"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 2);
    ASSERT_EQ(lmp->atom->nlocal, 2);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 3);

    x = lmp->atom->x;
    v = lmp->atom->v;
    q = lmp->atom->q;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(1)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(3)], -1.0, 1.0e-14);

    ASSERT_NEAR(lmp->atom->mass[1], 4.0, 1.0e-14);
    ASSERT_NEAR(lmp->atom->mass[2], 2.4, 1.0e-14);
    ASSERT_EQ(lmp->atom->mass_setflag[1], 1);
    ASSERT_EQ(lmp->atom->mass_setflag[2], 1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    lmp->input->one("change_box all triclinic");
    lmp->input->one("replicate 2 2 2 bbox");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->map_tag_max, 16);
    q = lmp->atom->q;
    ASSERT_NEAR(q[GETIDX(1)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(2)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(3)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(4)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(5)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(6)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(7)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(8)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(9)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(10)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(11)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(12)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(13)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(14)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(15)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(16)], -1.0, 1.0e-14);
}

TEST_F(AtomStyleTest, sphere)
{
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style sphere");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 1);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 0);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_EQ(lmp->atom->radius_flag, 1);
    ASSERT_EQ(lmp->atom->omega_flag, 1);
    ASSERT_EQ(lmp->atom->torque_flag, 1);
    ASSERT_EQ(lmp->atom->angmom_flag, 0);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_NE(lmp->atom->omega, nullptr);
    ASSERT_EQ(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_EQ(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 2 box");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("set atom 1 mass 4.0");
    lmp->input->one("set atom 2 mass 4.0");
    lmp->input->one("set atom 3 mass 2.4");
    lmp->input->one("set atom 4 mass 2.4");
    lmp->input->one("set atom 1 omega -0.5  0.1  0.1");
    lmp->input->one("set atom 2 omega  0.5 -0.1 -0.1");
    lmp->input->one("set atom 3 omega -1.0  0.0  0.0");
    lmp->input->one("set atom 4 omega  0.0  1.0  0.0");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("atom_style sphere");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("units real");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->radius_flag, 1);
    ASSERT_EQ(lmp->atom->omega_flag, 1);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    auto x     = lmp->atom->x;
    auto v     = lmp->atom->v;
    auto rmass = lmp->atom->rmass;
    auto omega = lmp->atom->omega;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(1)], 4.0, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4.0, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(1)][0], -0.5, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(1)][1], 0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(2)][0], 0.5, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(2)][1], -0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(3)][0], -1.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(4)][1], 1.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(4)][2], 0.0, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    EXPECT_THAT(std::string(lmp->atom->atom_style), Eq("atomic"));
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("replicate 1 1 2");
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("sphere"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 4);
    ASSERT_EQ(lmp->atom->nlocal, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 2);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 4);

    rmass = lmp->atom->rmass;
    omega = lmp->atom->omega;
    ASSERT_NEAR(rmass[GETIDX(1)], 4.0, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 4.0, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(1)][0], -0.5, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(1)][1], 0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(2)][0], -1.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(3)][0], -0.5, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(3)][1], 0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(3)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(4)][0], -1.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(omega[GETIDX(4)][2], 0.0, 1.0e-14);
}

TEST_F(AtomStyleTest, ellipsoid)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style ellipsoid");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 1);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 0);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_EQ(lmp->atom->radius_flag, 0);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 1);
    ASSERT_EQ(lmp->atom->angmom_flag, 1);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_EQ(lmp->atom->radius, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_EQ(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 3 box");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("create_atoms 3 single  2.0  2.0 -2.1");
    lmp->input->one("create_atoms 3 single  2.0 -2.0  2.1");
    lmp->input->one("set type 1 mass 4.0");
    lmp->input->one("set type 2 mass 2.4");
    lmp->input->one("set type 3 mass 4.4");
    lmp->input->one("set type 1 shape 1.0 1.0 1.0");
    lmp->input->one("set type 2 shape 3.0 0.8 1.1");
    lmp->input->one("set atom 1 quat 1.0 0.0 0.0  0.0");
    lmp->input->one("set atom 2 quat 0.0 1.0 0.0 90.0");
    lmp->input->one("set atom 3 quat 1.0 0.0 1.0 30.0");
    lmp->input->one("set atom 4 quat 1.0 1.0 1.0 60.0");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("atom_style ellipsoid");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("units real");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 1);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x         = lmp->atom->x;
    auto v         = lmp->atom->v;
    auto type      = lmp->atom->type;
    auto ellipsoid = lmp->atom->ellipsoid;
    auto rmass     = lmp->atom->rmass;
    auto avec      = (AtomVecEllipsoid *)lmp->atom->avec;
    auto bonus     = avec->bonus;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(4)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], sqrt(5.0 / 30.0), 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("comm_style tiled");
    lmp->input->one("replicate 1 1 2 bbox");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type      = lmp->atom->type;
    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    avec      = (AtomVecEllipsoid *)lmp->atom->avec;
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(9)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(11)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(12)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(9)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(11)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(12)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.25056280708573159, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    avec      = (AtomVecEllipsoid *)lmp->atom->avec;
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(4)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(8)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(8)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.25056280708573159, 1.0e-14);
}

TEST_F(AtomStyleTest, line)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("dimension 2");
    lmp->input->one("atom_style line");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 1);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 1);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 1);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_EQ(lmp->atom->radius_flag, 1);
    ASSERT_EQ(lmp->atom->omega_flag, 1);
    ASSERT_EQ(lmp->atom->torque_flag, 1);
    ASSERT_EQ(lmp->atom->angmom_flag, 0);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_NE(lmp->atom->omega, nullptr);
    ASSERT_EQ(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 3 box");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.0");
    lmp->input->one("create_atoms 1 single -2.0 -2.0  0.0");
    lmp->input->one("create_atoms 2 single  2.0  2.0  0.0");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.0");
    lmp->input->one("create_atoms 3 single  3.0  0.0  0.0");
    lmp->input->one("create_atoms 3 single  0.0 -3.0  0.0");
    lmp->input->one("set type 1 mass 4.0");
    lmp->input->one("set type 2 mass 2.4");
    lmp->input->one("set type 3 mass 4.4");
    lmp->input->one("set type 1 length 2.0");
    lmp->input->one("set type 2 length 3.0");
    lmp->input->one("set atom 1 theta 0.0");
    lmp->input->one("set atom 2 theta 90.0");
    lmp->input->one("set atom 3 theta 30.0");
    lmp->input->one("set atom 4 theta 60.0");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("dimension 2");
    lmp->input->one("atom_style line");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("units real");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->line_flag, 1);
    ASSERT_NE(lmp->atom->line, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x     = lmp->atom->x;
    auto v     = lmp->atom->v;
    auto type  = lmp->atom->type;
    auto line  = lmp->atom->line;
    auto rmass = lmp->atom->rmass;
    auto avec  = (AtomVecLine *)lmp->atom->avec;
    auto bonus = avec->bonus;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 3.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -3.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(line[GETIDX(1)], 0);
    ASSERT_EQ(line[GETIDX(2)], 1);
    ASSERT_EQ(line[GETIDX(3)], 2);
    ASSERT_EQ(line[GETIDX(4)], 3);
    ASSERT_EQ(line[GETIDX(5)], -1);
    ASSERT_EQ(line[GETIDX(6)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].length, 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].theta, 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].length, 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].theta, MathConst::MY_PI * 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].length, 3.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].theta, MathConst::MY_PI / 6.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].length, 3.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].theta, MathConst::MY_PI / 3.0, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("comm_style tiled");
    lmp->input->one("change_box all triclinic");
    lmp->input->one("replicate 1 2 1 bbox");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("line"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type  = lmp->atom->type;
    line  = lmp->atom->line;
    rmass = lmp->atom->rmass;
    avec  = (AtomVecLine *)lmp->atom->avec;
    bonus = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(line[GETIDX(1)], 0);
    ASSERT_EQ(line[GETIDX(3)], 1);
    ASSERT_EQ(line[GETIDX(5)], -1);
    ASSERT_EQ(line[GETIDX(6)], -1);
    ASSERT_EQ(line[GETIDX(7)], 2);
    ASSERT_EQ(line[GETIDX(9)], 3);
    ASSERT_EQ(line[GETIDX(11)], -1);
    ASSERT_EQ(line[GETIDX(12)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(9)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(11)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(12)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].length, 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].theta, 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].length, 3.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].theta, MathConst::MY_PI / 6.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].length, 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].theta, 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].length, 3.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].theta, MathConst::MY_PI / 6.0, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->nlines, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    line  = lmp->atom->line;
    rmass = lmp->atom->rmass;
    avec  = (AtomVecLine *)lmp->atom->avec;
    bonus = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(line[GETIDX(1)], 0);
    ASSERT_EQ(line[GETIDX(2)], -1);
    ASSERT_EQ(line[GETIDX(3)], 1);
    ASSERT_EQ(line[GETIDX(4)], -1);
    ASSERT_EQ(line[GETIDX(5)], 2);
    ASSERT_EQ(line[GETIDX(6)], -1);
    ASSERT_EQ(line[GETIDX(7)], 3);
    ASSERT_EQ(line[GETIDX(8)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(8)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].length, 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].theta, 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].length, 3.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].theta, MathConst::MY_PI / 6.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].length, 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].theta, 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].length, 3.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].theta, MathConst::MY_PI / 6.0, 1.0e-14);
}

TEST_F(AtomStyleTest, tri)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style tri");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 1);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 1);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 1);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_EQ(lmp->atom->radius_flag, 1);
    ASSERT_EQ(lmp->atom->omega_flag, 1);
    ASSERT_EQ(lmp->atom->torque_flag, 1);
    ASSERT_EQ(lmp->atom->angmom_flag, 1);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_NE(lmp->atom->omega, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_NE(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 3 box");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("create_atoms 3 single  2.0  2.0 -2.1");
    lmp->input->one("create_atoms 3 single  2.0 -2.0  2.1");
    lmp->input->one("set type 1 mass 4.0");
    lmp->input->one("set type 2 mass 2.4");
    lmp->input->one("set type 3 mass 4.4");
    lmp->input->one("set type 1 tri 1.0");
    lmp->input->one("set type 2 tri 1.5");
    lmp->input->one("set atom 1 quat 1.0 0.0 0.0  0.0");
    lmp->input->one("set atom 2 quat 0.0 1.0 0.0 90.0");
    lmp->input->one("set atom 3 quat 1.0 0.0 1.0 30.0");
    lmp->input->one("set atom 4 quat 1.0 1.0 1.0 60.0");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("atom_style tri");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("units real");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tri_flag, 1);
    ASSERT_NE(lmp->atom->tri, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x      = lmp->atom->x;
    auto v      = lmp->atom->v;
    auto type   = lmp->atom->type;
    auto tri    = lmp->atom->tri;
    auto rmass  = lmp->atom->rmass;
    auto radius = lmp->atom->radius;
    auto avec   = (AtomVecTri *)lmp->atom->avec;
    auto bonus  = avec->bonus;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(tri[GETIDX(1)], 0);
    ASSERT_EQ(tri[GETIDX(2)], 1);
    ASSERT_EQ(tri[GETIDX(3)], 2);
    ASSERT_EQ(tri[GETIDX(4)], 3);
    ASSERT_EQ(tri[GETIDX(5)], -1);
    ASSERT_EQ(tri[GETIDX(6)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(1)], 0.5773502691896258, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(2)], 0.5773502691896258, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(3)], 0.8660254037844390, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(4)], 0.8660254037844390, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(5)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(6)], 0.5, 1.0e-14);

    ASSERT_NEAR(bonus[0].inertia[0], 0.072258416330334363, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[1], 13.94589575227541, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[2], 14.017974903242481, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[0], 13.982119044342252, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[1], 0.10811427523057447, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[2], 13.945895752275419, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[0], 0.23541253382609079, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[1], 18.948744087979005, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[2], 19.15175691481879, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[0], 19.018309360029388, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[1], 0.36886008861549813, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[2], 18.948744087979012, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 0.92373678792937974, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0067268233964605153, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.016239988275423625, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], -0.38262430562330857, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.92420701665838023, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.023090752704586423, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0095209086022933816, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], -0.38107421822833848, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 0.90191325590647364, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.20096813128512517, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.10027584751231088, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.36892959143125964, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.8851587738551242, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.093233030525564045, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], -0.26735735980719261, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.36921753647848277, 1.0e-14);
    ASSERT_NEAR(bonus[0].c1[0], -0.14933690186163631, 1.0e-14);
    ASSERT_NEAR(bonus[0].c1[1], -0.55767753582520529, 1.0e-14);
    ASSERT_NEAR(bonus[0].c1[2], -0.0052525338293288905, 1.0e-14);
    ASSERT_NEAR(bonus[1].c1[0], 0.18561232929671426, 1.0e-14);
    ASSERT_NEAR(bonus[1].c1[1], -0.18681344121910506, 1.0e-14);
    ASSERT_NEAR(bonus[1].c1[2], 0.51379191773154387, 1.0e-14);
    ASSERT_NEAR(bonus[2].c1[0], -0.78775285695558617, 1.0e-14);
    ASSERT_NEAR(bonus[2].c1[1], -0.28372571379613187, 1.0e-14);
    ASSERT_NEAR(bonus[2].c1[2], -0.22123552085772166, 1.0e-14);
    ASSERT_NEAR(bonus[3].c1[0], -0.69845793336676587, 1.0e-14);
    ASSERT_NEAR(bonus[3].c1[1], -0.47901475403318056, 1.0e-14);
    ASSERT_NEAR(bonus[3].c1[2], 0.18083523090249506, 1.0e-14);
    ASSERT_NEAR(bonus[0].c2[0], 0.55733290519255385, 1.0e-14);
    ASSERT_NEAR(bonus[0].c2[1], 0.14942924536134222, 1.0e-14);
    ASSERT_NEAR(bonus[0].c2[2], 0.019602723119529659, 1.0e-14);
    ASSERT_NEAR(bonus[1].c2[0], 0.2208094914375279, 1.0e-14);
    ASSERT_NEAR(bonus[1].c2[1], -0.22223836695322471, 1.0e-14);
    ASSERT_NEAR(bonus[1].c2[2], -0.48496042114630089, 1.0e-14);
    ASSERT_NEAR(bonus[2].c2[0], 0.6304379562181005, 1.0e-14);
    ASSERT_NEAR(bonus[2].c2[1], -0.56674478453558153, 1.0e-14);
    ASSERT_NEAR(bonus[2].c2[2], 0.17705460333259249, 1.0e-14);
    ASSERT_NEAR(bonus[3].c2[0], 0.22007613459534958, 1.0e-14);
    ASSERT_NEAR(bonus[3].c2[1], 0.15093208974463554, 1.0e-14);
    ASSERT_NEAR(bonus[3].c2[2], -0.8238847002262436, 1.0e-14);
    ASSERT_NEAR(bonus[0].c3[0], -0.40799600333091751, 1.0e-14);
    ASSERT_NEAR(bonus[0].c3[1], 0.40824829046386302, 1.0e-14);
    ASSERT_NEAR(bonus[0].c3[2], -0.014350189290200811, 1.0e-14);
    ASSERT_NEAR(bonus[1].c3[0], -0.40642182073424188, 1.0e-14);
    ASSERT_NEAR(bonus[1].c3[1], 0.40905180817232945, 1.0e-14);
    ASSERT_NEAR(bonus[1].c3[2], -0.028831496585242929, 1.0e-14);
    ASSERT_NEAR(bonus[2].c3[0], 0.15731490073748589, 1.0e-14);
    ASSERT_NEAR(bonus[2].c3[1], 0.85047049833171351, 1.0e-14);
    ASSERT_NEAR(bonus[2].c3[2], 0.044180917525129149, 1.0e-14);
    ASSERT_NEAR(bonus[3].c3[0], 0.47838179877141634, 1.0e-14);
    ASSERT_NEAR(bonus[3].c3[1], 0.32808266428854471, 1.0e-14);
    ASSERT_NEAR(bonus[3].c3[2], 0.64304946932374851, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("change_box all triclinic");
    lmp->input->one("replicate 1 1 2");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("tri"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type   = lmp->atom->type;
    tri    = lmp->atom->tri;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    avec   = (AtomVecTri *)lmp->atom->avec;
    bonus  = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(tri[GETIDX(1)], 0);
    ASSERT_EQ(tri[GETIDX(3)], 1);
    ASSERT_EQ(tri[GETIDX(5)], -1);
    ASSERT_EQ(tri[GETIDX(6)], -1);
    ASSERT_EQ(tri[GETIDX(7)], 2);
    ASSERT_EQ(tri[GETIDX(9)], 3);
    ASSERT_EQ(tri[GETIDX(11)], -1);
    ASSERT_EQ(tri[GETIDX(12)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(9)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(11)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(12)], 4.4, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(1)], 0.5773502691896258, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(3)], 0.8660254037844390, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(5)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(6)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(7)], 0.5773502691896258, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(9)], 0.8660254037844390, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(11)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(12)], 0.5, 1.0e-14);

    ASSERT_NEAR(bonus[0].inertia[0], 0.072258416330334363, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[1], 13.94589575227541, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[2], 14.017974903242481, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[0], 0.23541253382609079, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[1], 18.948744087979005, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[2], 19.15175691481879, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[0], 0.072258416330334363, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[1], 13.94589575227541, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[2], 14.017974903242481, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[0], 0.23541253382609079, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[1], 18.948744087979005, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[2], 19.15175691481879, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 0.92373678792937974, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0067268233964605153, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.016239988275423625, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], -0.38262430562330857, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.90191325590647364, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.20096813128512517, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.10027584751231088, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.36892959143125964, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 0.92373678792937974, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.0067268233964605153, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.016239988275423625, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], -0.38262430562330857, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.90191325590647364, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.20096813128512517, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], 0.10027584751231088, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.36892959143125964, 1.0e-14);
    ASSERT_NEAR(bonus[0].c1[0], -0.14933690186163631, 1.0e-14);
    ASSERT_NEAR(bonus[0].c1[1], -0.55767753582520529, 1.0e-14);
    ASSERT_NEAR(bonus[0].c1[2], -0.0052525338293288905, 1.0e-14);
    ASSERT_NEAR(bonus[1].c1[0], -0.78775285695558617, 1.0e-14);
    ASSERT_NEAR(bonus[1].c1[1], -0.28372571379613187, 1.0e-14);
    ASSERT_NEAR(bonus[1].c1[2], -0.22123552085772166, 1.0e-14);
    ASSERT_NEAR(bonus[2].c1[0], -0.14933690186163631, 1.0e-14);
    ASSERT_NEAR(bonus[2].c1[1], -0.55767753582520529, 1.0e-14);
    ASSERT_NEAR(bonus[2].c1[2], -0.0052525338293288905, 1.0e-14);
    ASSERT_NEAR(bonus[3].c1[0], -0.78775285695558617, 1.0e-14);
    ASSERT_NEAR(bonus[3].c1[1], -0.28372571379613187, 1.0e-14);
    ASSERT_NEAR(bonus[3].c1[2], -0.22123552085772166, 1.0e-14);
    ASSERT_NEAR(bonus[0].c2[0], 0.55733290519255385, 1.0e-14);
    ASSERT_NEAR(bonus[0].c2[1], 0.14942924536134222, 1.0e-14);
    ASSERT_NEAR(bonus[0].c2[2], 0.019602723119529659, 1.0e-14);
    ASSERT_NEAR(bonus[1].c2[0], 0.6304379562181005, 1.0e-14);
    ASSERT_NEAR(bonus[1].c2[1], -0.56674478453558153, 1.0e-14);
    ASSERT_NEAR(bonus[1].c2[2], 0.17705460333259249, 1.0e-14);
    ASSERT_NEAR(bonus[2].c2[0], 0.55733290519255385, 1.0e-14);
    ASSERT_NEAR(bonus[2].c2[1], 0.14942924536134222, 1.0e-14);
    ASSERT_NEAR(bonus[2].c2[2], 0.019602723119529659, 1.0e-14);
    ASSERT_NEAR(bonus[3].c2[0], 0.6304379562181005, 1.0e-14);
    ASSERT_NEAR(bonus[3].c2[1], -0.56674478453558153, 1.0e-14);
    ASSERT_NEAR(bonus[3].c2[2], 0.17705460333259249, 1.0e-14);
    ASSERT_NEAR(bonus[0].c3[0], -0.40799600333091751, 1.0e-14);
    ASSERT_NEAR(bonus[0].c3[1], 0.40824829046386302, 1.0e-14);
    ASSERT_NEAR(bonus[0].c3[2], -0.014350189290200811, 1.0e-14);
    ASSERT_NEAR(bonus[1].c3[0], 0.15731490073748589, 1.0e-14);
    ASSERT_NEAR(bonus[1].c3[1], 0.85047049833171351, 1.0e-14);
    ASSERT_NEAR(bonus[1].c3[2], 0.044180917525129149, 1.0e-14);
    ASSERT_NEAR(bonus[2].c3[0], -0.40799600333091751, 1.0e-14);
    ASSERT_NEAR(bonus[2].c3[1], 0.40824829046386302, 1.0e-14);
    ASSERT_NEAR(bonus[2].c3[2], -0.014350189290200811, 1.0e-14);
    ASSERT_NEAR(bonus[3].c3[0], 0.15731490073748589, 1.0e-14);
    ASSERT_NEAR(bonus[3].c3[1], 0.85047049833171351, 1.0e-14);
    ASSERT_NEAR(bonus[3].c3[2], 0.044180917525129149, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->ntris, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    tri   = lmp->atom->tri;
    rmass = lmp->atom->rmass;
    avec  = (AtomVecTri *)lmp->atom->avec;
    bonus = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(tri[GETIDX(1)], 0);
    ASSERT_EQ(tri[GETIDX(2)], -1);
    ASSERT_EQ(tri[GETIDX(3)], 1);
    ASSERT_EQ(tri[GETIDX(4)], -1);
    ASSERT_EQ(tri[GETIDX(5)], 2);
    ASSERT_EQ(tri[GETIDX(6)], -1);
    ASSERT_EQ(tri[GETIDX(7)], 3);
    ASSERT_EQ(tri[GETIDX(8)], -1);
    ASSERT_NEAR(bonus[0].inertia[0], 0.072258416330334363, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[1], 13.94589575227541, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[2], 14.017974903242481, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[0], 0.23541253382609079, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[1], 18.948744087979005, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[2], 19.15175691481879, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[0], 0.072258416330334363, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[1], 13.94589575227541, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[2], 14.017974903242481, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[0], 0.23541253382609079, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[1], 18.948744087979005, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[2], 19.15175691481879, 1.0e-14);
}

TEST_F(AtomStyleTest, body_nparticle)
{
    if (!LAMMPS::is_installed_pkg("BODY")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style body nparticle 2 4");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));

    auto avec = (AtomVecBody *)lmp->atom->avec;
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_NE(avec->bptr, nullptr);
    ASSERT_THAT(std::string(avec->bptr->style), Eq("nparticle"));

    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 1);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 0);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_EQ(lmp->atom->radius_flag, 1);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 1);
    ASSERT_EQ(lmp->atom->angmom_flag, 1);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_EQ(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_EQ(lmp->atom->num_bond, nullptr);
    ASSERT_EQ(lmp->atom->bond_type, nullptr);
    ASSERT_EQ(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_EQ(lmp->atom->nspecial, nullptr);
    ASSERT_EQ(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 0);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    const char data_file[] = "\n4 atoms\n"
                             "4 bodies\n"
                             "3 atom types\n\n"
                             "-4.0 4.0 xlo xhi\n"
                             "-4.0 4.0 ylo yhi\n"
                             "-4.0 4.0 zlo zhi\n\n"
                             "Atoms # body\n\n"
                             "1 1 1 4.0 -2.0  2.0 0.1\n"
                             "2 1 1 4.0 -2.0 -2.0 -0.1\n"
                             "3 2 1 2.4  2.0  2.0 -0.1\n"
                             "4 2 1 2.4  2.0 -2.0 0.1\n\n"
                             "Bodies\n\n"
                             "1 1 12\n"
                             "2\n"
                             "2.0 2.0 0.0 0.0 0.0 0.0\n"
                             "-1.0 0.0 0.0\n"
                             "1.0 0.0 0.0\n"
                             "2 1 15\n"
                             "3\n"
                             "0.5 4.0 4.5 0.0 0.0 0.0\n"
                             "1.0 -0.5 0.0\n"
                             "1.0  0.5 0.0\n"
                             "-1.0 0.0 0.0\n"
                             "3 1 18\n"
                             "4\n"
                             "1.67188 1.46875 0.796875 0 0 0.546875\n"
                             "0.0  0.0 1.0\n"
                             "0.75 0.0 -0.25\n"
                             "0.5  0.5 -0.25\n"
                             "0.5 -0.5 -0.25\n"
                             "4 1 12\n"
                             "2\n"
                             "12.0 0.0 12.0 0.0 0.0 0.0\n"
                             "0.0 1.0 0.0\n"
                             "0.0 -3.0 0.0\n";
    FILE *fp = fopen("input_atom_styles.data", "w");
    fputs(data_file, fp);
    fclose(fp);
    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data input_atom_styles.data");
    lmp->input->one("create_atoms 3 single  2.0  2.0 -2.1");
    lmp->input->one("create_atoms 3 single  2.0 -2.0  2.1");
    lmp->input->one("set type 3 mass 4.4");
    lmp->input->one("set atom 1 quat 1.0 0.0 0.0  0.0");
    lmp->input->one("set atom 2 quat 0.0 1.0 0.0 90.0");
    lmp->input->one("set atom 3 quat 1.0 0.0 1.0 30.0");
    lmp->input->one("set atom 4 quat 1.0 1.0 1.0 60.0");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    auto x      = lmp->atom->x;
    auto v      = lmp->atom->v;
    auto type   = lmp->atom->type;
    auto body   = lmp->atom->body;
    auto rmass  = lmp->atom->rmass;
    auto radius = lmp->atom->radius;
    auto angmom = lmp->atom->angmom;
    auto bonus  = avec->bonus;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_NEAR(angmom[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(4)][2], 0.0, 1.0e-14);

    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(2)], 1);
    ASSERT_EQ(body[GETIDX(3)], 2);
    ASSERT_EQ(body[GETIDX(4)], 3);
    ASSERT_EQ(body[GETIDX(5)], -1);
    ASSERT_EQ(body[GETIDX(6)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(1)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(2)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(3)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(4)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(5)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(6)], 0.5, 1.0e-14);

    ASSERT_NEAR(bonus[0].inertia[0], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[1], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[1], 4.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[2], 4.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[0], 1.67188, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[1], 1.7746273249544022, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[2], 0.49099767504559777, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[0], 12.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[2], 12.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 1);
    ASSERT_EQ(bonus[2].ilocal, 2);
    ASSERT_EQ(bonus[3].ilocal, 3);
    ASSERT_EQ(bonus[0].ninteger, 1);
    ASSERT_EQ(bonus[1].ninteger, 1);
    ASSERT_EQ(bonus[2].ninteger, 1);
    ASSERT_EQ(bonus[3].ninteger, 1);
    ASSERT_EQ(bonus[0].ndouble, 6);
    ASSERT_EQ(bonus[1].ndouble, 9);
    ASSERT_EQ(bonus[2].ndouble, 12);
    ASSERT_EQ(bonus[3].ndouble, 6);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[1].ivalue, nullptr);
    ASSERT_NE(bonus[2].ivalue, nullptr);
    ASSERT_NE(bonus[3].ivalue, nullptr);
    ASSERT_EQ(bonus[0].ivalue[0], 2);
    ASSERT_EQ(bonus[1].ivalue[0], 3);
    ASSERT_EQ(bonus[2].ivalue[0], 4);
    ASSERT_EQ(bonus[3].ivalue[0], 2);
    ASSERT_NE(bonus[0].dvalue, nullptr);
    ASSERT_NE(bonus[1].dvalue, nullptr);
    ASSERT_NE(bonus[2].dvalue, nullptr);
    ASSERT_NE(bonus[3].dvalue, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("atom_style body nparticle 2 4");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("units real");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->body_flag, 1);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_NE(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);

    x      = lmp->atom->x;
    v      = lmp->atom->v;
    type   = lmp->atom->type;
    body   = lmp->atom->body;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    angmom = lmp->atom->angmom;
    avec   = (AtomVecBody *)lmp->atom->avec;
    bonus  = avec->bonus;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_NEAR(angmom[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(angmom[GETIDX(4)][2], 0.0, 1.0e-14);

    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(2)], 1);
    ASSERT_EQ(body[GETIDX(3)], 2);
    ASSERT_EQ(body[GETIDX(4)], 3);
    ASSERT_EQ(body[GETIDX(5)], -1);
    ASSERT_EQ(body[GETIDX(6)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(1)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(2)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(3)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(4)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(5)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(6)], 0.5, 1.0e-14);

    ASSERT_NEAR(bonus[0].inertia[0], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[1], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[0], 4.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[1], 4.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[0], 1.67188, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[1], 1.7746273249544022, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[2], 0.49099767504559777, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[0], 12.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[1], 12.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.25056280708573148, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.94328772332207422, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], -0.19726148412050853, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], -0.075803071317030998, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.25602748346874377, 1.0e-14);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 1);
    ASSERT_EQ(bonus[2].ilocal, 2);
    ASSERT_EQ(bonus[3].ilocal, 3);
    ASSERT_EQ(bonus[0].ninteger, 1);
    ASSERT_EQ(bonus[1].ninteger, 1);
    ASSERT_EQ(bonus[2].ninteger, 1);
    ASSERT_EQ(bonus[3].ninteger, 1);
    ASSERT_EQ(bonus[0].ndouble, 6);
    ASSERT_EQ(bonus[1].ndouble, 9);
    ASSERT_EQ(bonus[2].ndouble, 12);
    ASSERT_EQ(bonus[3].ndouble, 6);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[1].ivalue, nullptr);
    ASSERT_NE(bonus[2].ivalue, nullptr);
    ASSERT_NE(bonus[3].ivalue, nullptr);
    ASSERT_EQ(bonus[0].ivalue[0], 2);
    ASSERT_EQ(bonus[1].ivalue[0], 3);
    ASSERT_EQ(bonus[2].ivalue[0], 4);
    ASSERT_EQ(bonus[3].ivalue[0], 2);
    ASSERT_NE(bonus[0].dvalue, nullptr);
    ASSERT_NE(bonus[1].dvalue, nullptr);
    ASSERT_NE(bonus[2].dvalue, nullptr);
    ASSERT_NE(bonus[3].dvalue, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("comm_style tiled");
    lmp->input->one("replicate 1 1 2");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("body"));
    avec = (AtomVecBody *)lmp->atom->avec;
    ASSERT_THAT(std::string(avec->bptr->style), Eq("nparticle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 0);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type   = lmp->atom->type;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    body   = lmp->atom->body;
    bonus  = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(3)], 1);
    ASSERT_EQ(body[GETIDX(5)], -1);
    ASSERT_EQ(body[GETIDX(6)], -1);
    ASSERT_EQ(body[GETIDX(7)], 2);
    ASSERT_EQ(body[GETIDX(9)], 3);
    ASSERT_EQ(body[GETIDX(11)], -1);
    ASSERT_EQ(body[GETIDX(12)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(9)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(11)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(12)], 4.4, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(1)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(3)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(5)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(6)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(7)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(9)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(11)], 0.5, 1.0e-14);
    ASSERT_NEAR(radius[GETIDX(12)], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[0], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[1], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].inertia[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[0], 1.67188, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[1], 1.7746273249544022, 1.0e-14);
    ASSERT_NEAR(bonus[1].inertia[2], 0.49099767504559777, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[0], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[1], 2.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].inertia[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[0], 1.67188, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[1], 1.7746273249544022, 1.0e-14);
    ASSERT_NEAR(bonus[3].inertia[2], 0.49099767504559777, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.25056280708573148, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.25056280708573148, 1.0e-14);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 2);
    ASSERT_EQ(bonus[2].ilocal, 4);
    ASSERT_EQ(bonus[3].ilocal, 6);
    ASSERT_EQ(bonus[0].ninteger, 1);
    ASSERT_EQ(bonus[1].ninteger, 1);
    ASSERT_EQ(bonus[2].ninteger, 1);
    ASSERT_EQ(bonus[3].ninteger, 1);
    ASSERT_EQ(bonus[0].ndouble, 6);
    ASSERT_EQ(bonus[1].ndouble, 12);
    ASSERT_EQ(bonus[2].ndouble, 6);
    ASSERT_EQ(bonus[3].ndouble, 12);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[0].ivalue, nullptr);
    ASSERT_NE(bonus[1].ivalue, nullptr);
    ASSERT_NE(bonus[2].ivalue, nullptr);
    ASSERT_NE(bonus[3].ivalue, nullptr);
    ASSERT_EQ(bonus[0].ivalue[0], 2);
    ASSERT_EQ(bonus[1].ivalue[0], 4);
    ASSERT_EQ(bonus[2].ivalue[0], 2);
    ASSERT_EQ(bonus[3].ivalue[0], 4);
    ASSERT_NE(bonus[0].dvalue, nullptr);
    ASSERT_NE(bonus[1].dvalue, nullptr);
    ASSERT_NE(bonus[2].dvalue, nullptr);
    ASSERT_NE(bonus[3].dvalue, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->nbodies, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    body   = lmp->atom->body;
    rmass  = lmp->atom->rmass;
    radius = lmp->atom->radius;
    avec   = (AtomVecBody *)lmp->atom->avec;
    bonus  = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(body[GETIDX(1)], 0);
    ASSERT_EQ(body[GETIDX(2)], -1);
    ASSERT_EQ(body[GETIDX(3)], 1);
    ASSERT_EQ(body[GETIDX(4)], -1);
    ASSERT_EQ(body[GETIDX(5)], 2);
    ASSERT_EQ(body[GETIDX(6)], -1);
    ASSERT_EQ(body[GETIDX(7)], 3);
    ASSERT_EQ(body[GETIDX(8)], -1);
    ASSERT_EQ(bonus[0].ilocal, 0);
    ASSERT_EQ(bonus[1].ilocal, 2);
    ASSERT_EQ(bonus[2].ilocal, 4);
    ASSERT_EQ(bonus[3].ilocal, 6);
}

TEST_F(AtomStyleTest, bond)
{
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style bond");
    lmp->input->one("newton on");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 1);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 0);
    ASSERT_EQ(lmp->atom->radius_flag, 0);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 0);
    ASSERT_EQ(lmp->atom->angmom_flag, 0);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_EQ(lmp->atom->angmom, nullptr);
    ASSERT_EQ(lmp->atom->torque, nullptr);
    ASSERT_EQ(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_NE(lmp->atom->num_bond, nullptr);
    ASSERT_NE(lmp->atom->bond_type, nullptr);
    ASSERT_NE(lmp->atom->bond_atom, nullptr);
    ASSERT_EQ(lmp->atom->num_angle, nullptr);
    ASSERT_EQ(lmp->atom->angle_type, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom1, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom2, nullptr);
    ASSERT_EQ(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_NE(lmp->atom->nspecial, nullptr);
    ASSERT_NE(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 3);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 3 box bond/types 2 "
                    "extra/bond/per/atom 2 extra/special/per/atom 4");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("create_atoms 3 single  2.0  2.0 -2.1");
    lmp->input->one("create_atoms 3 single  2.0 -2.0  2.1");
    lmp->input->one("mass 1 4.0");
    lmp->input->one("mass 2 2.4");
    lmp->input->one("mass 3 4.4");
    lmp->input->one("bond_style zero");
    lmp->input->one("bond_coeff * 4.0");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("create_bonds single/bond 1 1 5");
    lmp->input->one("create_bonds single/bond 1 1 3");
    lmp->input->one("create_bonds single/bond 2 3 5");
    lmp->input->one("create_bonds single/bond 2 3 6");
    lmp->input->one("create_bonds single/bond 2 5 6");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    // change sign on a couple of bond types. this will be undone on writing
    // a data file and thus should be positive again after reading it
    lmp->atom->bond_type[GETIDX(1)][0] *= -1;
    lmp->atom->bond_type[GETIDX(5)][0] *= -1;

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("units real");
    lmp->input->one("newton off");
    lmp->input->one("atom_style bond");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("bond_style zero");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("bond_coeff * 4.0");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto num_bond  = lmp->atom->num_bond;
    auto bond_type = lmp->atom->bond_type;
    auto bond_atom = lmp->atom->bond_atom;

    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 3);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 3);
    ASSERT_EQ(num_bond[GETIDX(6)], 2);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(5)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][1], 2);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(5)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(5)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(6)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(6)][1], 5);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("clear");
    lmp->input->one("units real");
    lmp->input->one("atom_style bond");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("bond_style zero");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("bond_coeff * 4.0");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x    = lmp->atom->x;
    auto v    = lmp->atom->v;
    auto type = lmp->atom->type;
    num_bond  = lmp->atom->num_bond;
    bond_type = lmp->atom->bond_type;
    bond_atom = lmp->atom->bond_atom;

    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 1);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 2);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 6);

    // change sign on bonds again. they will be toggled back in the restart
    lmp->atom->bond_type[GETIDX(1)][0] *= -1;
    lmp->atom->bond_type[GETIDX(5)][0] *= -1;

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("replicate 1 1 2 bbox");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("bond"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type      = lmp->atom->type;
    num_bond  = lmp->atom->num_bond;
    bond_atom = lmp->atom->bond_atom;
    bond_type = lmp->atom->bond_type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(5)], 1);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(num_bond[GETIDX(7)], 2);
    ASSERT_EQ(num_bond[GETIDX(9)], 2);
    ASSERT_EQ(num_bond[GETIDX(11)], 1);
    ASSERT_EQ(num_bond[GETIDX(12)], 0);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 6);
    ASSERT_EQ(bond_atom[GETIDX(7)][0], 11);
    ASSERT_EQ(bond_atom[GETIDX(7)][1], 9);
    ASSERT_EQ(bond_atom[GETIDX(9)][0], 11);
    ASSERT_EQ(bond_atom[GETIDX(9)][1], 12);
    ASSERT_EQ(bond_atom[GETIDX(11)][0], 12);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(7)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(7)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(9)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(9)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(11)][0], 2);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("delete_bonds all bond 2");
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    type = lmp->atom->type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);

    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(4)], 1);
    ASSERT_EQ(num_bond[GETIDX(5)], 2);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(num_bond[GETIDX(7)], 2);
    ASSERT_EQ(num_bond[GETIDX(8)], 1);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 4);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 4);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_atom[GETIDX(4)][0], 2);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 8);
    ASSERT_EQ(bond_atom[GETIDX(5)][1], 7);
    ASSERT_EQ(bond_atom[GETIDX(7)][0], 8);
    ASSERT_EQ(bond_atom[GETIDX(7)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(8)][0], 6);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], -2);
    ASSERT_EQ(bond_type[GETIDX(3)][1], -2);
    ASSERT_EQ(bond_type[GETIDX(4)][0], -2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(5)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(7)][0], -2);
    ASSERT_EQ(bond_type[GETIDX(7)][1], -2);
    ASSERT_EQ(bond_type[GETIDX(8)][0], -2);
}

TEST_F(AtomStyleTest, angle)
{
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style angle");
    lmp->input->one("newton on");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 0);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 1);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 0);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 0);
    ASSERT_EQ(lmp->atom->radius_flag, 0);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 0);
    ASSERT_EQ(lmp->atom->angmom_flag, 0);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_EQ(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_EQ(lmp->atom->angmom, nullptr);
    ASSERT_EQ(lmp->atom->torque, nullptr);
    ASSERT_EQ(lmp->atom->radius, nullptr);
    ASSERT_EQ(lmp->atom->rmass, nullptr);
    ASSERT_EQ(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_NE(lmp->atom->num_bond, nullptr);
    ASSERT_NE(lmp->atom->bond_type, nullptr);
    ASSERT_NE(lmp->atom->bond_atom, nullptr);
    ASSERT_NE(lmp->atom->num_angle, nullptr);
    ASSERT_NE(lmp->atom->angle_type, nullptr);
    ASSERT_NE(lmp->atom->angle_atom1, nullptr);
    ASSERT_NE(lmp->atom->angle_atom2, nullptr);
    ASSERT_NE(lmp->atom->angle_atom3, nullptr);
    ASSERT_EQ(lmp->atom->num_dihedral, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_type, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_EQ(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_EQ(lmp->atom->num_improper, nullptr);
    ASSERT_EQ(lmp->atom->improper_type, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom1, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom2, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom3, nullptr);
    ASSERT_EQ(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_NE(lmp->atom->nspecial, nullptr);
    ASSERT_NE(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 3);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 3 box bond/types 2 angle/types 2 "
                    "extra/bond/per/atom 2 extra/angle/per/atom 1 "
                    "extra/special/per/atom 4");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("create_atoms 3 single  2.0  2.0 -2.1");
    lmp->input->one("create_atoms 3 single  2.0 -2.0  2.1");
    lmp->input->one("mass 1 4.0");
    lmp->input->one("mass 2 2.4");
    lmp->input->one("mass 3 4.4");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("bond_style zero");
    lmp->input->one("bond_coeff * 4.0");
    lmp->input->one("angle_style zero");
    lmp->input->one("angle_coeff * 90.0");
    lmp->input->one("create_bonds single/bond 1 1 3");
    lmp->input->one("create_bonds single/bond 1 1 5");
    lmp->input->one("create_bonds single/bond 2 3 5");
    lmp->input->one("create_bonds single/bond 2 3 6");
    lmp->input->one("create_bonds single/bond 2 5 6");
    lmp->input->one("create_bonds single/angle 1 1 3 5");
    lmp->input->one("create_bonds single/angle 2 3 5 6");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nangles, 2);
    ASSERT_EQ(lmp->atom->nangletypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    // change sign on a couple of angle types. this will be undone on writing
    // a data file and thus should be positive again after reading it
    lmp->atom->angle_type[GETIDX(3)][0] *= -1;
    lmp->atom->angle_type[GETIDX(5)][0] *= -1;

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("units real");
    lmp->input->one("newton off");
    lmp->input->one("atom_style angle");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("bond_style zero");
    lmp->input->one("angle_style zero");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("bond_coeff * 4.0");
    lmp->input->one("angle_coeff * 90.0");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto num_bond    = lmp->atom->num_bond;
    auto bond_type   = lmp->atom->bond_type;
    auto bond_atom   = lmp->atom->bond_atom;
    auto num_angle   = lmp->atom->num_angle;
    auto angle_type  = lmp->atom->angle_type;
    auto angle_atom1 = lmp->atom->angle_atom1;
    auto angle_atom2 = lmp->atom->angle_atom2;
    auto angle_atom3 = lmp->atom->angle_atom3;

    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 3);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 3);
    ASSERT_EQ(num_bond[GETIDX(6)], 2);
    ASSERT_EQ(bond_type[GETIDX(1)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(1)][1], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(3)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(3)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_type[GETIDX(5)][1], 2);
    ASSERT_EQ(bond_type[GETIDX(5)][2], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][0], 2);
    ASSERT_EQ(bond_type[GETIDX(6)][1], 2);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 1);
    ASSERT_EQ(bond_atom[GETIDX(5)][1], 3);
    ASSERT_EQ(bond_atom[GETIDX(5)][2], 6);
    ASSERT_EQ(bond_atom[GETIDX(6)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(6)][1], 5);
    ASSERT_EQ(num_angle[GETIDX(1)], 1);
    ASSERT_EQ(num_angle[GETIDX(2)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 2);
    ASSERT_EQ(num_angle[GETIDX(4)], 0);
    ASSERT_EQ(num_angle[GETIDX(5)], 2);
    ASSERT_EQ(num_angle[GETIDX(6)], 1);
    ASSERT_EQ(angle_type[GETIDX(1)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(3)][1], 2);
    ASSERT_EQ(angle_type[GETIDX(5)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(5)][1], 2);
    ASSERT_EQ(angle_type[GETIDX(6)][0], 2);
    ASSERT_EQ(angle_atom1[GETIDX(1)][0], 1);
    ASSERT_EQ(angle_atom2[GETIDX(1)][0], 3);
    ASSERT_EQ(angle_atom3[GETIDX(1)][0], 5);
    ASSERT_EQ(angle_atom1[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_atom2[GETIDX(3)][0], 3);
    ASSERT_EQ(angle_atom3[GETIDX(3)][0], 5);
    ASSERT_EQ(angle_atom1[GETIDX(3)][1], 3);
    ASSERT_EQ(angle_atom2[GETIDX(3)][1], 5);
    ASSERT_EQ(angle_atom3[GETIDX(3)][1], 6);
    ASSERT_EQ(angle_atom1[GETIDX(5)][0], 1);
    ASSERT_EQ(angle_atom2[GETIDX(5)][0], 3);
    ASSERT_EQ(angle_atom3[GETIDX(5)][0], 5);
    ASSERT_EQ(angle_atom1[GETIDX(5)][1], 3);
    ASSERT_EQ(angle_atom2[GETIDX(5)][1], 5);
    ASSERT_EQ(angle_atom3[GETIDX(5)][1], 6);
    ASSERT_EQ(angle_atom1[GETIDX(6)][0], 3);
    ASSERT_EQ(angle_atom2[GETIDX(6)][0], 5);
    ASSERT_EQ(angle_atom3[GETIDX(6)][0], 6);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("clear");
    lmp->input->one("units real");
    lmp->input->one("atom_style angle");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("bond_style zero");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("bond_coeff * 4.0");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x     = lmp->atom->x;
    auto v     = lmp->atom->v;
    auto type  = lmp->atom->type;
    num_bond   = lmp->atom->num_bond;
    bond_atom  = lmp->atom->bond_atom;
    num_angle  = lmp->atom->num_angle;
    angle_type = lmp->atom->angle_type;

    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(num_bond[GETIDX(1)], 2);
    ASSERT_EQ(num_bond[GETIDX(2)], 0);
    ASSERT_EQ(num_bond[GETIDX(3)], 2);
    ASSERT_EQ(num_bond[GETIDX(4)], 0);
    ASSERT_EQ(num_bond[GETIDX(5)], 1);
    ASSERT_EQ(num_bond[GETIDX(6)], 0);
    ASSERT_EQ(bond_atom[GETIDX(1)][0], 3);
    ASSERT_EQ(bond_atom[GETIDX(1)][1], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][0], 5);
    ASSERT_EQ(bond_atom[GETIDX(3)][1], 6);
    ASSERT_EQ(bond_atom[GETIDX(5)][0], 6);

    ASSERT_EQ(num_angle[GETIDX(1)], 0);
    ASSERT_EQ(num_angle[GETIDX(2)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 1);
    ASSERT_EQ(num_angle[GETIDX(4)], 0);
    ASSERT_EQ(num_angle[GETIDX(5)], 1);
    ASSERT_EQ(num_angle[GETIDX(6)], 0);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(5)][0], 2);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("replicate 1 1 2");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("angle"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type       = lmp->atom->type;
    num_angle  = lmp->atom->num_angle;
    angle_type = lmp->atom->angle_type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(num_angle[GETIDX(1)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 1);
    ASSERT_EQ(num_angle[GETIDX(5)], 1);
    ASSERT_EQ(num_angle[GETIDX(6)], 0);
    ASSERT_EQ(num_angle[GETIDX(7)], 0);
    ASSERT_EQ(num_angle[GETIDX(9)], 1);
    ASSERT_EQ(num_angle[GETIDX(11)], 1);
    ASSERT_EQ(num_angle[GETIDX(12)], 0);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(5)][0], 2);
    ASSERT_EQ(angle_type[GETIDX(9)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(11)][0], 2);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("delete_bonds all angle 2");
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    type = lmp->atom->type;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(num_angle[GETIDX(1)], 0);
    ASSERT_EQ(num_angle[GETIDX(2)], 0);
    ASSERT_EQ(num_angle[GETIDX(3)], 1);
    ASSERT_EQ(num_angle[GETIDX(4)], 1);
    ASSERT_EQ(num_angle[GETIDX(5)], 0);
    ASSERT_EQ(num_angle[GETIDX(6)], 0);
    ASSERT_EQ(num_angle[GETIDX(7)], 1);
    ASSERT_EQ(num_angle[GETIDX(8)], 1);
    ASSERT_EQ(angle_type[GETIDX(3)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(4)][0], -2);
    ASSERT_EQ(angle_type[GETIDX(7)][0], 1);
    ASSERT_EQ(angle_type[GETIDX(8)][0], -2);
}

TEST_F(AtomStyleTest, full_ellipsoid)
{
    if (!LAMMPS::is_installed_pkg("ASPHERE")) GTEST_SKIP();
    if (!LAMMPS::is_installed_pkg("MOLECULE")) GTEST_SKIP();

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("atom_style hybrid full ellipsoid");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    auto hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("full"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("ellipsoid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);
    ASSERT_EQ(lmp->atom->natoms, 0);
    ASSERT_EQ(lmp->atom->nlocal, 0);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nmax, 1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->nellipsoids, 0);
    ASSERT_EQ(lmp->atom->nlines, 0);
    ASSERT_EQ(lmp->atom->ntris, 0);
    ASSERT_EQ(lmp->atom->nbodies, 0);
    ASSERT_EQ(lmp->atom->nbonds, 0);
    ASSERT_EQ(lmp->atom->nangles, 0);
    ASSERT_EQ(lmp->atom->ndihedrals, 0);
    ASSERT_EQ(lmp->atom->nimpropers, 0);
    ASSERT_EQ(lmp->atom->ntypes, 0);
    ASSERT_EQ(lmp->atom->nbondtypes, 0);
    ASSERT_EQ(lmp->atom->nangletypes, 0);
    ASSERT_EQ(lmp->atom->ndihedraltypes, 0);
    ASSERT_EQ(lmp->atom->nimpropertypes, 0);
    ASSERT_EQ(lmp->atom->bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->improper_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_bond_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_angle_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_dihedral_per_atom, 0);
    ASSERT_EQ(lmp->atom->extra_improper_per_atom, 0);

    ASSERT_EQ(lmp->atom->sphere_flag, 0);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 1);
    ASSERT_EQ(lmp->atom->line_flag, 0);
    ASSERT_EQ(lmp->atom->tri_flag, 0);
    ASSERT_EQ(lmp->atom->body_flag, 0);
    ASSERT_EQ(lmp->atom->peri_flag, 0);
    ASSERT_EQ(lmp->atom->electron_flag, 0);
    ASSERT_EQ(lmp->atom->wavepacket_flag, 0);
    ASSERT_EQ(lmp->atom->sph_flag, 0);
    ASSERT_EQ(lmp->atom->molecule_flag, 1);
    ASSERT_EQ(lmp->atom->molindex_flag, 0);
    ASSERT_EQ(lmp->atom->molatom_flag, 0);
    ASSERT_EQ(lmp->atom->q_flag, 1);
    ASSERT_EQ(lmp->atom->mu_flag, 0);
    ASSERT_EQ(lmp->atom->rmass_flag, 1);
    ASSERT_EQ(lmp->atom->radius_flag, 0);
    ASSERT_EQ(lmp->atom->omega_flag, 0);
    ASSERT_EQ(lmp->atom->torque_flag, 1);
    ASSERT_EQ(lmp->atom->angmom_flag, 1);
    ASSERT_EQ(lmp->atom->vfrac_flag, 0);
    ASSERT_EQ(lmp->atom->spin_flag, 0);
    ASSERT_EQ(lmp->atom->eradius_flag, 0);
    ASSERT_EQ(lmp->atom->ervel_flag, 0);
    ASSERT_EQ(lmp->atom->erforce_flag, 0);
    ASSERT_EQ(lmp->atom->cs_flag, 0);
    ASSERT_EQ(lmp->atom->csforce_flag, 0);
    ASSERT_EQ(lmp->atom->vforce_flag, 0);
    ASSERT_EQ(lmp->atom->ervelforce_flag, 0);
    ASSERT_EQ(lmp->atom->etag_flag, 0);
    ASSERT_EQ(lmp->atom->rho_flag, 0);
    ASSERT_EQ(lmp->atom->esph_flag, 0);
    ASSERT_EQ(lmp->atom->cv_flag, 0);
    ASSERT_EQ(lmp->atom->vest_flag, 0);
    ASSERT_EQ(lmp->atom->dpd_flag, 0);
    ASSERT_EQ(lmp->atom->edpd_flag, 0);
    ASSERT_EQ(lmp->atom->tdpd_flag, 0);
    ASSERT_EQ(lmp->atom->mesont_flag, 0);
    ASSERT_EQ(lmp->atom->sp_flag, 0);
    ASSERT_EQ(lmp->atom->x0_flag, 0);
    ASSERT_EQ(lmp->atom->smd_flag, 0);
    ASSERT_EQ(lmp->atom->damage_flag, 0);
    ASSERT_EQ(lmp->atom->contact_radius_flag, 0);
    ASSERT_EQ(lmp->atom->smd_data_9_flag, 0);
    ASSERT_EQ(lmp->atom->smd_stress_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_flag, 0);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate_flag, 0);
    ASSERT_EQ(lmp->atom->pdscale, 1.0);

    ASSERT_NE(lmp->atom->tag, nullptr);
    ASSERT_NE(lmp->atom->type, nullptr);
    ASSERT_NE(lmp->atom->mask, nullptr);
    ASSERT_NE(lmp->atom->image, nullptr);
    ASSERT_NE(lmp->atom->x, nullptr);
    ASSERT_NE(lmp->atom->v, nullptr);
    ASSERT_NE(lmp->atom->f, nullptr);
    ASSERT_NE(lmp->atom->q, nullptr);
    ASSERT_EQ(lmp->atom->mu, nullptr);
    ASSERT_EQ(lmp->atom->omega, nullptr);
    ASSERT_NE(lmp->atom->angmom, nullptr);
    ASSERT_NE(lmp->atom->torque, nullptr);
    ASSERT_EQ(lmp->atom->radius, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_EQ(lmp->atom->line, nullptr);
    ASSERT_EQ(lmp->atom->tri, nullptr);
    ASSERT_EQ(lmp->atom->body, nullptr);
    ASSERT_NE(lmp->atom->molecule, nullptr);
    ASSERT_EQ(lmp->atom->molindex, nullptr);
    ASSERT_EQ(lmp->atom->molatom, nullptr);
    ASSERT_NE(lmp->atom->num_bond, nullptr);
    ASSERT_NE(lmp->atom->bond_type, nullptr);
    ASSERT_NE(lmp->atom->bond_atom, nullptr);
    ASSERT_NE(lmp->atom->num_angle, nullptr);
    ASSERT_NE(lmp->atom->angle_type, nullptr);
    ASSERT_NE(lmp->atom->angle_atom1, nullptr);
    ASSERT_NE(lmp->atom->angle_atom2, nullptr);
    ASSERT_NE(lmp->atom->angle_atom3, nullptr);
    ASSERT_NE(lmp->atom->num_dihedral, nullptr);
    ASSERT_NE(lmp->atom->dihedral_type, nullptr);
    ASSERT_NE(lmp->atom->dihedral_atom1, nullptr);
    ASSERT_NE(lmp->atom->dihedral_atom2, nullptr);
    ASSERT_NE(lmp->atom->dihedral_atom3, nullptr);
    ASSERT_NE(lmp->atom->dihedral_atom4, nullptr);
    ASSERT_NE(lmp->atom->num_improper, nullptr);
    ASSERT_NE(lmp->atom->improper_type, nullptr);
    ASSERT_NE(lmp->atom->improper_atom1, nullptr);
    ASSERT_NE(lmp->atom->improper_atom2, nullptr);
    ASSERT_NE(lmp->atom->improper_atom3, nullptr);
    ASSERT_NE(lmp->atom->improper_atom4, nullptr);
    ASSERT_EQ(lmp->atom->maxspecial, 1);
    ASSERT_NE(lmp->atom->nspecial, nullptr);
    ASSERT_NE(lmp->atom->special, nullptr);
    ASSERT_EQ(lmp->atom->vfrac, nullptr);
    ASSERT_EQ(lmp->atom->s0, nullptr);
    ASSERT_EQ(lmp->atom->x0, nullptr);
    ASSERT_EQ(lmp->atom->sp, nullptr);
    ASSERT_EQ(lmp->atom->fm, nullptr);
    ASSERT_EQ(lmp->atom->fm_long, nullptr);
    ASSERT_EQ(lmp->atom->spin, nullptr);
    ASSERT_EQ(lmp->atom->eradius, nullptr);
    ASSERT_EQ(lmp->atom->ervel, nullptr);
    ASSERT_EQ(lmp->atom->erforce, nullptr);
    ASSERT_EQ(lmp->atom->ervelforce, nullptr);
    ASSERT_EQ(lmp->atom->cs, nullptr);
    ASSERT_EQ(lmp->atom->csforce, nullptr);
    ASSERT_EQ(lmp->atom->vforce, nullptr);
    ASSERT_EQ(lmp->atom->etag, nullptr);
    ASSERT_EQ(lmp->atom->uCond, nullptr);
    ASSERT_EQ(lmp->atom->uMech, nullptr);
    ASSERT_EQ(lmp->atom->uChem, nullptr);
    ASSERT_EQ(lmp->atom->uCG, nullptr);
    ASSERT_EQ(lmp->atom->uCGnew, nullptr);
    ASSERT_EQ(lmp->atom->duChem, nullptr);
    ASSERT_EQ(lmp->atom->dpdTheta, nullptr);
    ASSERT_EQ(lmp->atom->cc, nullptr);
    ASSERT_EQ(lmp->atom->cc_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_temp, nullptr);
    ASSERT_EQ(lmp->atom->edpd_flux, nullptr);
    ASSERT_EQ(lmp->atom->edpd_cv, nullptr);
    ASSERT_EQ(lmp->atom->length, nullptr);
    ASSERT_EQ(lmp->atom->buckling, nullptr);
    ASSERT_EQ(lmp->atom->bond_nt, nullptr);
    ASSERT_EQ(lmp->atom->contact_radius, nullptr);
    ASSERT_EQ(lmp->atom->smd_data_9, nullptr);
    ASSERT_EQ(lmp->atom->smd_stress, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain, nullptr);
    ASSERT_EQ(lmp->atom->eff_plastic_strain_rate, nullptr);
    ASSERT_EQ(lmp->atom->damage, nullptr);
    ASSERT_EQ(lmp->atom->rho, nullptr);
    ASSERT_EQ(lmp->atom->drho, nullptr);
    ASSERT_EQ(lmp->atom->esph, nullptr);
    ASSERT_EQ(lmp->atom->desph, nullptr);
    ASSERT_EQ(lmp->atom->cv, nullptr);
    ASSERT_EQ(lmp->atom->vest, nullptr);
    ASSERT_EQ(lmp->atom->nmolecule, 0);
    ASSERT_EQ(lmp->atom->molecules, nullptr);
    ASSERT_EQ(lmp->atom->nivector, 0);
    ASSERT_EQ(lmp->atom->ndvector, 0);
    ASSERT_EQ(lmp->atom->iname, nullptr);
    ASSERT_EQ(lmp->atom->dname, nullptr);
    ASSERT_EQ(lmp->atom->mass, nullptr);
    ASSERT_EQ(lmp->atom->mass_setflag, nullptr);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);
    ASSERT_EQ(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->map_style, 3);
    ASSERT_EQ(lmp->atom->map_user, 0);
    ASSERT_EQ(lmp->atom->map_tag_max, -1);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("create_box 3 box bond/types 2 "
                    "extra/bond/per/atom 2 extra/special/per/atom 4");
    lmp->input->one("create_atoms 1 single -2.0  2.0  0.1");
    lmp->input->one("create_atoms 1 single -2.0 -2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0  2.0 -0.1");
    lmp->input->one("create_atoms 2 single  2.0 -2.0  0.1");
    lmp->input->one("create_atoms 3 single  2.0  2.0 -2.1");
    lmp->input->one("create_atoms 3 single  2.0 -2.0  2.1");
    lmp->input->one("set type 1 mass 4.0");
    lmp->input->one("set type 2 mass 2.4");
    lmp->input->one("set type 3 mass 4.4");
    lmp->input->one("mass 1 4.0");
    lmp->input->one("mass 2 2.4");
    lmp->input->one("mass 3 4.4");
    lmp->input->one("set type 1 shape 1.0 1.0 1.0");
    lmp->input->one("set type 2 shape 3.0 0.8 1.1");
    lmp->input->one("set atom 1 quat 1.0 0.0 0.0  0.0");
    lmp->input->one("set atom 2 quat 0.0 1.0 0.0 90.0");
    lmp->input->one("set atom 3 quat 1.0 0.0 1.0 30.0");
    lmp->input->one("set atom 4 quat 1.0 1.0 1.0 60.0");
    lmp->input->one("set atom 1 charge -0.5");
    lmp->input->one("set atom 2 charge  0.5");
    lmp->input->one("set atom 3 charge -1.0");
    lmp->input->one("set atom 4 charge  1.0");
    lmp->input->one("set atom 5 charge  2.0");
    lmp->input->one("set atom 6 charge -2.0");
    lmp->input->one("bond_style zero");
    lmp->input->one("bond_coeff * 4.0");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("create_bonds single/bond 1 1 5");
    lmp->input->one("create_bonds single/bond 1 1 3");
    lmp->input->one("create_bonds single/bond 2 3 5");
    lmp->input->one("create_bonds single/bond 2 3 6");
    lmp->input->one("create_bonds single/bond 2 5 6");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nbonds, 5);
    ASSERT_EQ(lmp->atom->nbondtypes, 2);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->nextra_grow, 0);
    ASSERT_EQ(lmp->atom->nextra_restart, 0);
    ASSERT_EQ(lmp->atom->nextra_border, 0);
    ASSERT_EQ(lmp->atom->nextra_grow_max, 0);
    ASSERT_EQ(lmp->atom->nextra_restart_max, 0);
    ASSERT_EQ(lmp->atom->nextra_border_max, 0);
    ASSERT_EQ(lmp->atom->nextra_store, 0);
    ASSERT_EQ(lmp->atom->extra_grow, nullptr);
    ASSERT_EQ(lmp->atom->extra_restart, nullptr);
    ASSERT_EQ(lmp->atom->extra_border, nullptr);
    ASSERT_EQ(lmp->atom->extra, nullptr);

    ASSERT_NE(lmp->atom->mass, nullptr);
    ASSERT_NE(lmp->atom->rmass, nullptr);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->mass_setflag, nullptr);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("write_data test_atom_styles.data nocoeff");
    lmp->input->one("clear");
    lmp->input->one("units real");
    lmp->input->one("atom_style hybrid full ellipsoid");
    lmp->input->one("pair_style zero 4.0");
    lmp->input->one("bond_style zero");
    lmp->input->one("atom_modify map array");
    lmp->input->one("read_data test_atom_styles.data");
    lmp->input->one("pair_coeff * *");
    lmp->input->one("bond_coeff * 4.0");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    ASSERT_NE(lmp->atom->avec, nullptr);
    hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("full"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("ellipsoid"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);

    ASSERT_EQ(lmp->atom->natoms, 6);
    ASSERT_EQ(lmp->atom->nlocal, 6);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->ellipsoid_flag, 1);
    ASSERT_NE(lmp->atom->ellipsoid, nullptr);
    ASSERT_NE(lmp->atom->sametag, nullptr);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_style, 1);
    ASSERT_EQ(lmp->atom->map_user, 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 6);

    auto x         = lmp->atom->x;
    auto v         = lmp->atom->v;
    auto q         = lmp->atom->q;
    auto type      = lmp->atom->type;
    auto ellipsoid = lmp->atom->ellipsoid;
    auto rmass     = lmp->atom->rmass;

    auto avec  = (AtomVecEllipsoid *)hybrid->styles[1];
    auto bonus = avec->bonus;
    ASSERT_NEAR(x[GETIDX(1)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(1)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][0], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(2)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(3)][2], -0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(4)][2], 0.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][1], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(5)][2], -2.1, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][0], 2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][1], -2.0, 1.0e-14);
    ASSERT_NEAR(x[GETIDX(6)][2], 2.1, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(1)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(2)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(3)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(4)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(5)][2], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][0], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][1], 0.0, 1.0e-14);
    ASSERT_NEAR(v[GETIDX(6)][2], 0.0, 1.0e-14);
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_NEAR(q[GETIDX(1)], -0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(2)], 0.5, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(3)], -1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(4)], 1.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(5)], 2.0, 1.0e-14);
    ASSERT_NEAR(q[GETIDX(6)], -2.0, 1.0e-14);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(4)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], sqrt(0.5), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], sqrt(5.0 / 30.0), 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], sqrt(5.0 / 30.0), 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("pair_coeff * *");
    lmp->input->one("group two id 2:4:2");
    lmp->input->one("delete_atoms group two compress no");
    lmp->input->one("write_restart test_atom_styles.restart");
    lmp->input->one("clear");
    lmp->input->one("read_restart test_atom_styles.restart");
    lmp->input->one("replicate 1 1 2 bbox");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_THAT(std::string(lmp->atom->atom_style), Eq("hybrid"));
    hybrid = (AtomVecHybrid *)lmp->atom->avec;
    ASSERT_EQ(hybrid->nstyles, 2);
    ASSERT_THAT(std::string(hybrid->keywords[0]), Eq("full"));
    ASSERT_THAT(std::string(hybrid->keywords[1]), Eq("ellipsoid"));
    ASSERT_NE(hybrid->styles[0], nullptr);
    ASSERT_NE(hybrid->styles[1], nullptr);
    ASSERT_NE(lmp->atom->avec, nullptr);
    ASSERT_EQ(lmp->atom->natoms, 8);
    ASSERT_EQ(lmp->atom->nlocal, 8);
    ASSERT_EQ(lmp->atom->nghost, 0);
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_NE(lmp->atom->nmax, -1);
    ASSERT_EQ(lmp->atom->tag_enable, 1);
    ASSERT_EQ(lmp->atom->molecular, 1);
    ASSERT_EQ(lmp->atom->ntypes, 3);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 0);
    ASSERT_EQ(lmp->atom->map_tag_max, 12);

    type      = lmp->atom->type;
    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    avec      = (AtomVecEllipsoid *)hybrid->styles[1];
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(5)], 3);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 1);
    ASSERT_EQ(type[GETIDX(9)], 2);
    ASSERT_EQ(type[GETIDX(11)], 3);
    ASSERT_EQ(type[GETIDX(12)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(9)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(11)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(12)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(9)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(11)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(12)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.25056280708573159, 1.0e-14);

    if (!verbose) ::testing::internal::CaptureStdout();
    lmp->input->one("reset_atom_ids");
    if (!verbose) ::testing::internal::GetCapturedStdout();
    ASSERT_EQ(lmp->atom->nellipsoids, 4);
    ASSERT_EQ(lmp->atom->tag_consecutive(), 1);
    ASSERT_EQ(lmp->atom->map_tag_max, 8);

    ellipsoid = lmp->atom->ellipsoid;
    rmass     = lmp->atom->rmass;
    hybrid    = (AtomVecHybrid *)lmp->atom->avec;
    avec      = (AtomVecEllipsoid *)hybrid->styles[1];
    bonus     = avec->bonus;
    ASSERT_EQ(type[GETIDX(1)], 1);
    ASSERT_EQ(type[GETIDX(2)], 3);
    ASSERT_EQ(type[GETIDX(3)], 2);
    ASSERT_EQ(type[GETIDX(4)], 3);
    ASSERT_EQ(type[GETIDX(5)], 1);
    ASSERT_EQ(type[GETIDX(6)], 3);
    ASSERT_EQ(type[GETIDX(7)], 2);
    ASSERT_EQ(type[GETIDX(8)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(1)], 0);
    ASSERT_EQ(ellipsoid[GETIDX(2)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(3)], 1);
    ASSERT_EQ(ellipsoid[GETIDX(4)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(5)], 2);
    ASSERT_EQ(ellipsoid[GETIDX(6)], -1);
    ASSERT_EQ(ellipsoid[GETIDX(7)], 3);
    ASSERT_EQ(ellipsoid[GETIDX(8)], -1);
    ASSERT_NEAR(rmass[GETIDX(1)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(2)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(3)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(4)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(5)], 4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(6)], 4.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(7)], 2.4, 1.0e-14);
    ASSERT_NEAR(rmass[GETIDX(8)], 4.4, 1.0e-14);

    ASSERT_NEAR(bonus[0].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[0].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[1].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[0], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[1], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[2].shape[2], 0.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[0], 1.5, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[1], 0.4, 1.0e-14);
    ASSERT_NEAR(bonus[3].shape[2], 0.55, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[0].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[1].quat[3], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[0], 1.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[1], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[2].quat[3], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[0], 0.9351131265310294, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[1], 0.25056280708573159, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[2], 0.0, 1.0e-14);
    ASSERT_NEAR(bonus[3].quat[3], 0.25056280708573159, 1.0e-14);
}

} // namespace LAMMPS_NS

int main(int argc, char **argv)
{
    MPI_Init(&argc, &argv);
    ::testing::InitGoogleMock(&argc, argv);

    // handle arguments passed via environment variable
    if (const char *var = getenv("TEST_ARGS")) {
        auto env = split_words(var);
        for (auto arg : env) {
            if (arg == "-v") {
                verbose = true;
            }
        }
    }
    if ((argc > 1) && (strcmp(argv[1], "-v") == 0)) verbose = true;

    int rv = RUN_ALL_TESTS();
    MPI_Finalize();
    return rv;
}
