#!/usr/bin/env python3.3
# -*- coding: utf-8 -*-
#
# Simple web interface to cve-search to display the last entries
# and view a specific CVE.
#
# Software is free software released under the "Modified BSD license"
#

# Copyright (c) 2013-2014 	Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2014-2016 	Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# imports
import os
import sys
_runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(_runPath, ".."))

from tornado.wsgi import WSGIContainer
from tornado.httpserver import HTTPServer
from tornado.ioloop import IOLoop
from flask import Flask, render_template, request, redirect, jsonify, send_file, abort
from flask.ext.login import LoginManager, current_user, login_user, logout_user, login_required
from passlib.hash import pbkdf2_sha256
from redis import exceptions as redisExceptions

import jinja2
import json
import re
import argparse
import time
import urllib
import random
import signal
import logging
import subprocess
from io import TextIOWrapper, BytesIO
from logging.handlers import RotatingFileHandler

from lib.User import User
from lib.Config import Configuration
from lib.PluginManager import PluginManager
from lib.Toolkit import toStringFormattedCPE, toOldCPE, currentTime, isURL, vFeedName, convertDateToDBFormat, mergeSearchResults
import lib.CVEs as cves
import lib.DatabaseLayer as db
from sbin.db_whitelist import *
from sbin.db_blacklist import *

# parse command line arguments
argparser = argparse.ArgumentParser(description='Start CVE-Search web component')
argparser.add_argument('-v', action='store_true', help='verbose output')
args = argparser.parse_args()

# variables
app = Flask(__name__, static_folder='static', static_url_path='/static')
app.config['MONGO_DBNAME'] = Configuration.getMongoDB()
app.config['SECRET_KEY'] = str(random.getrandbits(256))
pageLength = Configuration.getPageLength()
plugManager = PluginManager()

# login manager
login_manager = LoginManager()
login_manager.init_app(app)
# db connectors
redisdb = Configuration.getRedisVendorConnection()

# functions
def getBrowseList(vendor):
    result = {}
    if (vendor is None) or type(vendor) == list:
        v1 = redisdb.smembers("t:/o")
        v2 = redisdb.smembers("t:/a")
        v3 = redisdb.smembers("t:/h")
        vendor = sorted(list(set(list(v1) + list(v2) + list(v3))))
        cpe = None
    else:
        cpenum = redisdb.scard("v:" + vendor)
        if cpenum < 1:
            return page_not_found(404)
        p = redisdb.smembers("v:" + vendor)
        cpe = sorted(list(p))
    result["vendor"] = vendor
    result["product"] = cpe
    return result


def whitelist_mark(cve):
    whitelistitems = compile(db.getRules('whitelist'))
    # ensures we're working with a list object, in case we get a pymongo.cursor object
    cve = list(cve)
    # check the cpes (full or partially) in the whitelist
    for cveid in cve:
        cpes = cveid['vulnerable_configuration']
        for c in cpes:
            if any(regex.match(c) for regex in whitelistitems):
                cve[cve.index(cveid)]['whitelisted'] = 'yes'
    return cve


def blacklist_mark(cve):
    blacklistitems = compile(db.getRules('blacklist'))
    # ensures we're working with a list object, in case we get a pymongo.cursor object
    cve = list(cve)
    # check the cpes (full or partially) in the blacklist
    for cveid in cve:
        cpes = cveid['vulnerable_configuration']
        for c in cpes:
            if any(regex.match(c) for regex in blacklistitems):
                cve[cve.index(cveid)]['blacklisted'] = 'yes'
    return cve


def compile(regexes):
  r=[]
  for rule in regexes:
    r.append(re.compile(rule))
  return r


def addCPEToList(cpe, listType, cpeType=None):
    if not cpeType:
        cpeType='cpe'
    if listType.lower() in ("blacklist", "black", "b", "bl"):
        if insertBlacklist(cpe,cpeType):
            return True
        else:
            return False
    if listType.lower() in ("whitelist", "white", "w", "wl"):
        if insertWhitelist(cpe,cpeType):
            return True
        else:
            return False


def getVersionsOfProduct(product):
    p = redisdb.smembers("p:" + product)
    return sorted(list(p))


def adminStats():
    return db.getDBStats()


def filterUpdateField(data):
    if not data: return data
    returnvalue = []
    for line in data.split("\n"):
        print(line)
        if (not line.startswith("[+]Success to create index") and
            not line == "Not modified" and
            not line.startswith("Starting")):
                returnvalue.append(line)
    return "\n".join(returnvalue)


def filter_logic(f, limit, skip):
    query = []
    # retrieving lists
    if f['blacklistSelect'] == "on":
        regexes = db.getRules('blacklist')
        if len(regexes) != 0:
            exp = "^(?!" + "|".join(regexes) + ")"
            query.append({'$or': [{'vulnerable_configuration': re.compile(exp)},
                                  {'vulnerable_configuration': {'$exists': False}},
                                  {'vulnerable_configuration': []}
                                  ]})
    if f['whitelistSelect'] == "hide":
        regexes = db.getRules('whitelist')
        if len(regexes) != 0:
            exp = "^(?!" + "|".join(regexes) + ")"
            query.append({'$or': [{'vulnerable_configuration': re.compile(exp)},
                                  {'vulnerable_configuration': {'$exists': False}},
                                  {'vulnerable_configuration': []}
                                  ]})
    if f['unlistedSelect'] == "hide":
        wlregexes = compile(db.getRules('whitelist'))
        blregexes = compile(db.getRules('blacklist'))
        query.append({'$or': [{'vulnerable_configuration': {'$in': wlregexes}},
                              {'vulnerable_configuration': {'$in': blregexes}}]})
    if f['rejectedSelect'] == "hide":
        exp = "^(?!\*\* REJECT \*\*\s+DO NOT USE THIS CANDIDATE NUMBER.*)"
        query.append({'summary': re.compile(exp)})

    # plugin filters
    query.extend(plugManager.doFilter(f, current_user=current_user))

    # cvss logic
    if f['cvssSelect'] == "above":    query.append({'cvss': {'$gt': float(f['cvss'])}})
    elif f['cvssSelect'] == "equals": query.append({'cvss': float(f['cvss'])})
    elif f['cvssSelect'] == "below":  query.append({'cvss': {'$lt': float(f['cvss'])}})
    
    # date logic
    if f['timeSelect'] != "all":
        startDate = convertDateToDBFormat(f['startDate'])
        endDate = convertDateToDBFormat(f['endDate'])
        if f['timeSelect'] == "from":
            query.append({f['timeTypeSelect']: {'$gt': f['startDate']}})
        if f['timeSelect'] == "until":
            query.append({f['timeTypeSelect']: {'$lt': f['endDate']}})
        if f['timeSelect'] == "between":
            query.append({f['timeTypeSelect']: {'$gt': f['startDate'], '$lt': f['endDate']}})
        if f['timeSelect'] == "outside":
            query.append({'$or': [{f['timeTypeSelect']: {'$lt': f['startDate']}}, {f['timeTypeSelect']: {'$gt': f['endDate']}}]})
    cve=db.getCVEs(limit=limit, skip=skip, query=query)
    # marking relevant records
    if f['whitelistSelect'] == "on":   cve = whitelist_mark(cve)
    if f['blacklistSelect'] == "mark": cve = blacklist_mark(cve)
    plugManager.mark(cve, current_user=current_user)
    cve = list(cve)
    return cve


def markCPEs(cve):
    blacklist = compile(db.getRules('blacklist'))
    whitelist = compile(db.getRules('whitelist'))

    for conf in cve['vulnerable_configuration']:
        conf['list'] = 'none'
        conf['match'] = 'none'
        for w in whitelist:
            if w.match(conf['id']):
                conf['list'] = 'white'
                conf['match'] = w
        for b in blacklist:
            if b.match(conf['id']):
                conf['list'] = 'black'
                conf['match'] = b
    return cve


def getFilterSettingsFromPost(r):
    filters = dict(request.form)
    filters = {x: filters[x][0] for x in filters.keys()}
    # retrieving data
    try:
      cve = filter_logic(filters, pageLength, r)
    except:
      cve = db.getCVEs(limit=pageLength, skip=r)
    return(filters,cve)

@login_manager.user_loader
def load_user(id):
    return User.get(id)

# routes
@app.route('/')
def index():
    # get default page on HTTP get (navigating to page)
    filters={'blacklistSelect': 'on', 'whitelistSelect': 'on', 
             'unlistedSelect': 'show', 'timeSelect': 'all', 
             'startDate': '', 'endDate': '', 'timeTypeSelect': 'Modified',
             'cvssSelect': 'all', 'cvss': '', 'rejectedSelect': 'hide'}
    
    cve = filter_logic(filters, pageLength, 0)
    return render_template('index.html', cve=cve, r=0, pageLength=pageLength, filters=plugManager.getFilters())

@app.route('/', methods=['POST'])
def filterPost():
    settings,cve = getFilterSettingsFromPost(0)
    return render_template('index.html', settings=settings, cve=cve, r=0, pageLength=pageLength, filters=plugManager.getFilters())


@app.route('/r/<int:r>', methods=['POST'])
def filterLast(r):
    if not r:
        r = 0
    settings,cve = getFilterSettingsFromPost(r)
    return render_template('index.html', settings=settings, cve=cve, r=r, pageLength=pageLength, filters=plugManager.getFilters())

# Plugins
@app.route('/_get_plugins', methods=['GET'])
def get_plugins():
    if not current_user.is_authenticated(): # Don't show plugins requiring auth if not authenticated
        plugins = [{"name": x.getName(), "link": x.getUID()} for x in plugManager.getWebPluginsWithPage() if not x.requiresAuth]
    else:
        plugins = [{"name": x.getName(), "link": x.getUID()} for x in plugManager.getWebPluginsWithPage()]
    return jsonify({"plugins": plugins})

@app.route('/plugin/_get_cve_actions', methods=['GET'])
def get_cve_actions():
    cve = request.args.get('cve', type=str)
    if not current_user.is_authenticated(): # Don't show actions requiring auth if not authenticated
        actions = [x for x in plugManager.getCVEActions(cve, current_user=current_user) if not x['auth']]
    else:
        actions = plugManager.getCVEActions(cve, current_user=current_user)
    return jsonify({"actions": actions})

@app.route('/plugin/<plugin>', methods=['GET'])
def openPlugin(plugin):
    if plugManager.requiresAuth(plugin) and not current_user.is_authenticated():
        return render_template("requiresAuth.html")
    else:
        page, args = plugManager.openPage(plugin, current_user=current_user)
        if page:
            try:
                return render_template(page, **args)
            except jinja2.exceptions.TemplateSyntaxError: return render_template("error.html", status={'except': 'plugin-page-corrupt'})
            except jinja2.exceptions.TemplateNotFound:    return render_template("error.html", status={'except': 'plugin-page-not-found', 'page': page})
        else: abort(404)

@app.route('/plugin/<plugin>/subpage/<page>', methods=['GET'])
def openPluginSubpage(plugin, page):
    if plugManager.requiresAuth(plugin) and not current_user.is_authenticated():
        return render_template("requiresAuth.html")
    else:
        page, args = plugManager.openSubpage(plugin, page, current_user=current_user)
        if page:
            try:
                return render_template(page, **args)
            except jinja2.exceptions.TemplateSyntaxError: return render_template("error.html", status={'except': 'plugin-page-corrupt'})
            except jinja2.exceptions.TemplateNotFound:    return render_template("error.html", status={'except': 'plugin-page-not-found', 'page': page})
        else: abort(404)

@app.route('/plugin/<plugin>/_cve_action/<action>', methods=['GET'])
def jsonCVEAction(plugin, action):
    cve = request.args.get('cve', type=str).split(",")
    if plugManager.onCVEAction(cve, plugin, action, current_user=current_user):
        return jsonify({'status': 'plugin_action_complete'})
    else:
        return jsonify({'status': 'plugin_action_failed'})

# API
@app.route('/api/cpe2.3/<path:cpe>', methods=['GET'])
def cpe23(cpe):
    cpe = toStringFormattedCPE(cpe)
    if not cpe: cpe='None'
    return cpe

@app.route('/api/cpe2.2/<path:cpe>', methods=['GET'])
def cpe22(cpe):
    cpe = toOldCPE(cpe)
    if not cpe: cpe='None'
    return cpe

@app.route('/api/cvefor/<path:cpe>', methods=['GET'])
def apiCVEFor(cpe):
    cpe=urllib.parse.unquote_plus(cpe)
    cpe=toStringFormattedCPE(cpe)
    r = []
    cvesp = cves.last(rankinglookup=False, namelookup=False, vfeedlookup=True, capeclookup=False)
    for x in db.cvesForCPE(cpe):
        r.append(cvesp.getcve(x['id']))
    return json.dumps(r)

@app.route('/api/cve/<cveid>', methods=['GET'])
def apiCVE(cveid):
    cvesp = cves.last(rankinglookup=True, namelookup=True, vfeedlookup=True, capeclookup=True)
    cve = cvesp.getcve(cveid=cveid)
    if cve is None:
        cve = {}
    return (jsonify(cve))

@app.route('/api/browse/<vendor>', methods=['GET'])
@app.route('/api/browse/', methods=['GET'])
@app.route('/api/browse', methods=['GET'])
def apibrowse(vendor=None):
    if vendor is not None:
        vendor = urllib.parse.quote_plus(vendor).lower()
    browseList = getBrowseList(vendor)
    if isinstance(browseList, dict):
        return (jsonify(browseList))
    else:
        return (jsonify({}))

@app.route('/api/search/<vendor>/<path:product>', methods=['GET'])
def apisearch(vendor=None, product=None):
    if vendor is None or product is None:
        return (jsonify({}))
    search = vendor + ":" + product
    return (json.dumps(db.cvesForCPE(search)))

@app.route('/api/dbInfo', methods=['GET'])
def apidbInfo():
    return (json.dumps(db.getDBStats()))

@app.route('/cve/<cveid>')
def cve(cveid):
    cveid = cveid.upper()
    cvesp = cves.last(rankinglookup=True, namelookup=True, vfeedlookup=True, capeclookup=True,subscorelookup=True)
    cve = cvesp.getcve(cveid=cveid)
    if cve is None:
        return render_template('error.html',status={'except':'cve-not-found','info':{'cve':cveid}})
    cve = markCPEs(cve)
    
    plugManager.onCVEOpen(cveid, current_user=current_user)
    pluginData = plugManager.cvePluginInfo(cveid, current_user=current_user)
    return render_template('cve.html', cve=cve, plugins=pluginData)

@app.route('/browse/<vendor>')
@app.route('/browse/')
def browse(vendor=None):
    try:
        if vendor is not None:
            vendor = urllib.parse.quote_plus(vendor).lower()
        browseList = getBrowseList(vendor)
        vendor = browseList["vendor"]
        product = browseList["product"]
        return render_template('browse.html', product=product, vendor=vendor)
    except redisExceptions.ConnectionError:
        return render_template('error.html',
                               status={'except':'redis-connection',
                                       'info':{'host':Configuration.getRedisHost(),'port':Configuration.getRedisPort()}})

@app.route('/cwe')
def cwe():
    cwes=[x for x in db.getCWEs() if x["weaknessabs"].lower()=="class"]
    #cwes=db.getCWEs()
    return render_template('cwe.html', cwes=cwes, capec=None)

@app.route('/cwe/<cweid>')
def relatedCWE(cweid):
    cwes={x["id"]: x["name"] for x in db.getCWEs()}
    return render_template('cwe.html', cwes=cwes, cwe=cweid, capec=db.getCAPECFor(cweid))

@app.route('/capec/<capecid>')
def capec(capecid):
    cwes={x["id"]: x["name"] for x in db.getCWEs()}
    return render_template('capec.html', cwes=cwes, capec=db.getCAPEC(capecid))

@app.route('/search', methods=['POST'])
def searchText():
    search = request.form.get('search')
    dbResults = db.getSearchResults(search)
    plugResults = plugManager.getSearchResults(search)
    result = mergeSearchResults(dbResults, plugResults)
    cve=result['data']
    errors=result['errors'] if 'errors' in result else []
    return render_template('search.html', cve=cve, errors=errors)

@app.route('/search/<vendor>/<path:product>')
def search(vendor=None, product=None):
    search = vendor + ":" + product
    cve = db.cvesForCPE(search)
    return render_template('search.html', vendor=vendor, product=product, cve=cve)

@app.route('/link/<vFeedMap>/<field>/<path:value>')
def link(vFeedMap=None,field=None,value=None):
    vFeedMap=htmlDedode(vFeedMap)
    field=htmlDedode(field)
    value=htmlDedode(value)
    search="%s.%s"%(vFeedMap,field)
    regex = re.compile(re.escape(value), re.I)
    cve=db.vFeedLinked(search, regex)
    # marking relevant records
    cve = whitelist_mark(cve)
    cve = blacklist_mark(cve)
    plugManager.mark(cve, current_user=current_user)
    cve = list(cve)
    cvssList=[float(x['cvss']) for x in cve if 'cvss' in x]
    if cvssList:
        stats={'maxCVSS': max(cvssList), 'minCVSS': min(cvssList),'count':len(cve)}
    else:
        stats={'maxCVSS': 0, 'minCVSS': 0, 'count':len(cve)}
    return render_template('linked.html', vFeedMap=vFeedMap, field=field, value=value, cve=cve, stats=stats)

@app.route('/admin')
@app.route('/admin/')
def admin():
    if Configuration.loginRequired():
        if not current_user.is_authenticated():
            return render_template('login.html')
    else:
        person = User.get("_dummy_")
        login_user(person)
    output = None
    if os.path.isfile(Configuration.getUpdateLogFile()):
        with open(Configuration.getUpdateLogFile()) as updateFile:
            separator="==========================\n"
            output=updateFile.read().split(separator)[-2:]
            output=separator+separator.join(output)
    return render_template('admin.html', status="default", stats=adminStats(), updateOutput=filterUpdateField(output))


@app.route('/admin/updatedb')
@login_required
def updatedb():
    process = subprocess.Popen(["python3", os.path.join(_runPath, "../sbin/db_updater.py"), "-civ"], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    out, err = process.communicate()
    output="%s\n\nErrors:\n%s"%(str(out,'utf-8'),str(err,'utf-8')) if err else str(out,'utf-8')
    return jsonify({"updateOutput": output, "status": "db_updated"})

@app.route('/admin/whitelist/import', methods=['POST'])
@login_required
def whitelistImport(force=None, path=None):
    file = request.files['file']
    force = request.form.get('force')
    count = countWhitelist()
    if (count == 0) | (not count) | (force == "f"):
        dropWhitelist()
        importWhitelist(TextIOWrapper(file.stream))
        status = "wl_imported"
    else:
        status = "wl_already_filled"
    return render_template('admin.html', status=status, stats=adminStats())

@app.route('/admin/whitelist/export')
@login_required
def whitelistExport(force=None, path=None):
    bytIO = BytesIO()
    bytIO.write(bytes(exportWhitelist(), "utf-8"))
    bytIO.seek(0)
    return send_file(bytIO, as_attachment=True, attachment_filename="whitelist.txt")


@app.route('/admin/whitelist/drop')
@login_required
def whitelistDrop():
    dropWhitelist()
    return jsonify({"status":"wl_dropped"})


@app.route('/admin/whitelist')
@login_required
def whitelistView():
    return render_template('list.html', rules=db.getWhitelist(), listType="Whitelist")


@app.route('/admin/addToList')
@login_required
def listAdd(): 
    cpe = request.args.get('cpe')
    cpeType = request.args.get('type')
    lst = request.args.get('list')
    if cpe and cpeType and lst:
        status = "added_to_list" if addCPEToList(cpe, lst, cpeType) else "already_exists_in_list"
        returnList = db.getWhitelist() if lst=="whitelist" else db.getBlacklist()
        return jsonify({"status":status, "rules":returnList, "listType":lst.title()})
    else: return jsonify({"status": "could_not_add_to_list"})


@app.route('/admin/removeFromList')
@login_required
def listRemove():
    cpe = request.args.get('cpe', type=str)
    cpe = urllib.parse.quote_plus(cpe).lower()
    cpe = cpe.replace("%3a", ":")
    cpe = cpe.replace("%2f", "/")
    lst = request.args.get('list', type=str)
    if cpe and lst:
        result=removeWhitelist(cpe) if lst.lower()=="whitelist" else removeBlacklist(cpe)
        status = "removed_from_list" if (result > 0) else "already_removed_from_list"
    else:
        status = "invalid_cpe"
    returnList = db.getWhitelist() if lst=="whitelist" else db.getBlacklist()
    return jsonify({"status":status, "rules":returnList, "listType":lst.title()})


@app.route('/admin/editInList')
@login_required
def listEdit():
    oldCPE = request.args.get('oldCPE')
    newCPE = request.args.get('cpe')
    lst = request.args.get('list')
    CPEType = request.args.get('type')
    if oldCPE and newCPE:
        result = updateWhitelist(oldCPE, newCPE, CPEType) if lst=="whitelist" else updateBlacklist(oldCPE, newCPE, CPEType)
        status = "cpelist_updated" if (result) else "cpelist_update_failed"
    else:
        status = "invalid_cpe"
    returnList = list(db.getWhitelist()) if lst=="whitelist" else list(db.getBlacklist())
    return jsonify({"rules":returnList, "status":status, "listType":lst})


@app.route('/admin/blacklist/import')
@login_required
def blacklistImport():
    file = request.files['file']
    force = request.form.get('force')
    count = countBlacklist()
    if (count == 0) | (not count) | (force == "f"):
        dropBlacklist()
        importBlacklist(TextIOWrapper(file.stream))
        status = "bl_imported"
    else:
        status = "bl_already_filled"
    return render_template('admin.html', status=status, stats=adminStats())

@app.route('/admin/blacklist/export')
@login_required
def blacklistExport():
    bytIO = BytesIO()
    bytIO.write(bytes(exportBlacklist(), "utf-8"))
    bytIO.seek(0)
    return send_file(bytIO, as_attachment=True, attachment_filename="blacklist.txt")

@app.route('/admin/blacklist/drop')
@login_required
def blacklistDrop():
    dropBlacklist()
    return jsonify({status:"bl_dropped"})


@app.route('/admin/blacklist')
@login_required
def blacklistView():
    return render_template('list.html', rules=db.getBlacklist(), listType="Blacklist")


@app.route('/admin/listmanagement/add')
@login_required
def listManagementAdd():
    # retrieve the separate item parts
    item = request.args.get('item', type=str)
    listType = request.args.get('list', type=str)

    vendor = None
    product = None
    version = None
    pattern = re.compile('^[a-z:/0-9.~_%-]+$')

    if pattern.match(item):
        item = item.split(":")
        added = False
        if len(item) == 1:
            # only vendor, so a check on cpe type is needed
            if redisdb.sismember("t:/o", item[0]):
                if addCPEToList("cpe:/o:" + item[0], listType):
                    added = True
            if redisdb.sismember("t:/a", item[0]):
                if addCPEToList("cpe:/a:" + item[0], listType):
                    added = True
            if redisdb.sismember("t:/h", item[0]):
                if addCPEToList("cpe:/h:" + item[0], listType):
                    added = True
            browseList = getBrowseList(None)
            vendor = browseList['vendor']
        elif 4 > len(item) > 1:
            # cpe type can be found with a mongo regex query
            result = db.getCVEs(query={'cpe_2_2': {'$regex': item[1]}})
            if result.count() != 0:
                prefix = ((result[0])['cpe_2_2'])[:7]
                if len(item) == 2:
                    if addCPEToList(prefix + item[0] + ":" + item[1], listType):
                        added = True
                if len(item) == 3:
                    if addCPEToList(prefix + item[0] + ":" + item[1] + ":" + item[2], listType):
                        added = True
            vendor = item[0]
        if len(item) > 2:
            product = item[1]
            version = getVersionsOfProduct(product)
        else:
            product = (getBrowseList(vendor))['product']
        status = "added_to_list" if added else "could_not_add_to_list"
    else:
        browseList = getBrowseList(None)
        vendor = browseList['vendor']
        status = "invalid_cpe"
    j={"status":status, "listType":listType}
    return jsonify(j)


@app.route('/admin/listmanagement/<vendor>/<product>')
@app.route('/admin/listmanagement/<vendor>')
@app.route('/admin/listmanagement')
@login_required
def listManagement(vendor=None, product=None):
    try:
        if product is None:
            # no product selected yet, so same function as /browse can be used
            if vendor:
                vendor = urllib.parse.quote_plus(vendor).lower()
            browseList = getBrowseList(vendor)
            vendor = browseList["vendor"]
            product = browseList["product"]
            version = None
        else:
            # product selected, product versions required
            version = getVersionsOfProduct(urllib.parse.quote_plus(product).lower())
        return render_template('listmanagement.html', vendor=vendor, product=product, version=version)
    except redisExceptions.ConnectionError:
        return render_template('error.html',
                               status={'except':'redis-connection',
                                       'info':{'host':Configuration.getRedisHost(),'port':Configuration.getRedisPort()}})


@app.route('/login', methods=['post'])
def login_check():
    # validate username and password
    username = request.form.get('username')
    password = request.form.get('password')
    person = User.get(username)
    try:
        if person and pbkdf2_sha256.verify(password, person.password):
            login_user(person)
            return render_template('admin.html', status="logged_in", stats=adminStats())
        else:
            return render_template('login.html', status="wrong_user_pass")
    except:
        return render_template('login.html', status="outdated_database")


@app.route('/logout')
def logout():
    logout_user()
    return redirect("/")

# error handeling
@app.errorhandler(404)
def page_not_found(e):
    return render_template('404.html'), 404


# filters
@app.template_filter('currentTime')
def currentTimeFilter(utc):
    return currentTime(utc)

@app.template_filter('htmlDecode')
def htmlDedode(string):
    return urllib.parse.unquote_plus(string)

@app.template_filter('htmlEncode')
def htmlEncode(string):
    return urllib.parse.quote_plus(string).lower()

@app.template_filter('isURL')
def isURLFilter(string):
    return isURL(string)

@app.template_filter('vFeedName')
def vFeedNameFilter(string):
    return vFeedName(string)

@app.template_filter('sortIntLikeStr')
def sortIntLikeStrFilter(datalist):
    return sorted(datalist, key=lambda k: int(k))

# signal handlers
def sig_handler(sig, frame):
    print('Caught signal: %s' % sig)
    IOLoop.instance().add_callback(shutdown)


def shutdown():
    MAX_WAIT_SECONDS_BEFORE_SHUTDOWN = 3
    print('Stopping http server')
    http_server.stop()

    print('Will shutdown in %s seconds ...' % MAX_WAIT_SECONDS_BEFORE_SHUTDOWN)
    io_loop = IOLoop.instance()
    deadline = time.time() + MAX_WAIT_SECONDS_BEFORE_SHUTDOWN

    def stop_loop():
        now = time.time()
        if now < deadline and (io_loop._callbacks or io_loop._timeouts):
            io_loop.add_timeout(now + 1, stop_loop)
        else:
            io_loop.stop()
            print('Shutdown')
    stop_loop()

if __name__ == '__main__':
    # get properties
    flaskHost = Configuration.getFlaskHost()
    flaskPort = Configuration.getFlaskPort()
    flaskDebug = Configuration.getFlaskDebug()
    # logging
    if Configuration.getLogging():
        logfile = Configuration.getLogfile()
        pathToLog = os.path.join(_runPath, logfile.rsplit('/', 1)[0])
        #pathToLog = os.path.join(_runPath, pathToLog)
        logfile = os.path.join(_runPath, logfile)
        if not os.path.exists(pathToLog):
            os.makedirs(pathToLog)
        maxLogSize = Configuration.getMaxLogSize()
        backlog = Configuration.getBacklog()
        file_handler = RotatingFileHandler(logfile, maxBytes=maxLogSize, backupCount=backlog)
        file_handler.setLevel(logging.ERROR)
        formatter = logging.Formatter("%(asctime)s - %(name)s - %(levelname)s - %(message)s")
        file_handler.setFormatter(formatter)
        app.logger.addHandler(file_handler)

    plugManager.loadPlugins()

    if flaskDebug:
        # start debug flask server
        app.run(host=flaskHost, port=flaskPort, debug=flaskDebug)
    else:
        # start asynchronous server using tornado wrapper for flask
        # ssl connection
        print("Server starting...")
        if Configuration.useSSL():
            cert = Configuration.getSSLCert()
            key = Configuration.getSSLKey()
            ssl_options = {"certfile": cert,
                           "keyfile": key}
        else:
            ssl_options = None
        signal.signal(signal.SIGTERM, sig_handler)
        signal.signal(signal.SIGINT, sig_handler)
        global http_server
        http_server = HTTPServer(WSGIContainer(app), ssl_options=ssl_options)
        http_server.bind(flaskPort, address=flaskHost)
        http_server.start(0)  # Forks multiple sub-processes
        IOLoop.instance().start()
