#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# Minimal web interface to cve-search to display the last entries
# and view a specific CVE.
#
# Software is free software released under the "Modified BSD license"
#

# Copyright (c) 2013-2016  Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2014-2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# imports
import os
import sys
import urllib
_runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(_runPath, ".."))

from flask import render_template, request

import lib.Toolkit       as tk

from lib.Config  import Configuration
from web.api     import API, APIError

class Minimal(API):
    #############
    # Variables #
    #############
    defaultFilters={'timeSelect': 'all',
                    'startDate': '', 'endDate': '', 'timeTypeSelect': 'Modified',
                    'cvssSelect': 'all', 'cvss': '', 'rejectedSelect': 'hide'}
    args = {'pageLength':   Configuration.getPageLength(),
            'listLogin':    Configuration.listLoginRequired(),
            'minimal':      True}

    def __init__(self):
        self.minimal = True
        super().__init__()
        filters = [{'n': 'htmlEncode',      'f': self.htmlEncode},
                   {'n': 'htmlDecode',      'f': self.htmlDecode},
                   {'n': 'sortCWE',         'f': self.sortCWE}]
        context_processors = [self.JSON2HTMLTable]
        error_handlers = [{'e': 404, 'f': self.page_not_found}]

        for _filter in filters:            self.addFilter(_filter)
        for context in context_processors: self.addContextProcessors(context)
        for handler in error_handlers:     self.app.register_error_handler(handler['e'], handler['f'])

    #############
    # Functions #
    #############
    def addFilter(self, _filter):
        self.app.add_template_filter(_filter['f'], _filter['n'])

    def addContextProcessors(self, context_processor):
        self.app.context_processor(context_processor)

    def getFilterSettingsFromPost(self, r):
        filters = dict(request.form)
        filters = {x: filters[x][0] for x in filters.keys()}
        errors  = False
        # retrieving data
        try:
            cve = self.filter_logic(filters, r)
        except Exception as e:
            cve = self.db.CVE.query(limit=self.args['pageLength'], skip=r)
            errors = True
        return {'settings': filters, 'cve': cve, 'errors': errors}

    ##########
    # ROUTES #
    ##########
    # /
    def index(self):
        cve = self.filter_logic(self.defaultFilters, 0)
        return render_template('index.html', cve=cve, r=0, **self.args)

    # /
    def index_post(self):
        args = dict(self.getFilterSettingsFromPost(0), **self.args)
        return render_template('index.html', r=0, **args)

    # /r/<r>
    def index_filter_get(self, r):
        if not r or r < 0: r = 0
        cve = self.filter_logic(self.defaultFilters, r)
        return render_template('index.html', cve=cve, r=r, **self.args)

    # /r/<r>
    def index_filter_post(self, r):
        if not r or r < 0: r = 0
        args = dict(self.getFilterSettingsFromPost(r), **self.args)
        return render_template('index.html', r=r, **args)

    # /cve/<cveid>
    def cve(self, cveid):
        cve = self.api_cve(cveid)
        if not cve:
            return render_template('error.html',status={'except':'cve-not-found','info':{'cve':cveid}},minimal=self.minimal)
        return render_template('cve.html', cve=cve, minimal=self.minimal)

    # /cwe
    def cwe(self):
        cwes=[x for x in self.api_cwe() if x.weakness.lower()=="class"]
        return render_template('cwe.html', cwes=cwes, capec=None, minimal=self.minimal)

    # /cwe/<cweid>
    def relatedCWE(self, cweid):
        return render_template('cwe.html', cwe=cweid, capec=self.db.CAPEC.relatedTo(cweid), minimal=self.minimal)

    # /capec/<capecid>
    def capec(self, capecid):
        return render_template('capec.html', capec=self.db.CAPEC.get(capecid), minimal=self.minimal)

    # /browse
    # /browse/
    # /browse/<vendor>
    def browse(self, vendor=None):
        try:
            data = self.api_browse(vendor)
            if 'product' in data and 'vendor' in data:
                return render_template('browse.html', product=data["product"], vendor=data["vendor"], minimal=self.minimal)
            else:
                return render_template('error.html', minimal=self.minimal, status={'except':'browse_exception', 'info': 'No CPE'})
        except APIError as e:
            return render_template('error.html', minimal=self.minimal, status={'except':'browse_exception', 'info':e.message})

    # /search/<vendor>/<product>
    def search(self, vendor=None, product=None):
        cve = self.api_search(vendor, product)
        return render_template('search.html', vendor=vendor, product=product, cve=cve, minimal=self.minimal)

    # /search
    def freetext_search(self):
        search = request.form.get('search')
        result = self.api_text_search(search)
        cve=result['data']
        errors=result['errors'] if 'errors' in result else []
        return render_template('search.html', cve=cve, errors=errors, minimal=self.minimal)

    # /link/<key>/<value>
    def link(self, key=None,value=None):
        cve   = self.api_link(key, value)
        stats = cve.pop("stats")
        cve = cve['cves']
        return render_template('linked.html', via4map=key.split(".")[0], field='.'.join(key.split(".")[1:]),
                               value=value, cve=cve, stats=stats, minimal=self.minimal)


    ###########
    # Filters #
    ###########
    def htmlEncode(self, string):
        return urllib.parse.quote_plus(string).lower()
 
    def htmlDecode(self, string):
        return urllib.parse.unquote_plus(string)

    def sortCWE(self, datalist):
        return sorted(datalist, key=lambda k: k.id)

    def JSON2HTMLTable(self):
        # Doublequote, because we have to |safe the content for the tags
        def doublequote(data):
          return urllib.parse.quote_plus(urllib.parse.quote_plus(data))

        def JSON2HTMLTableFilter(data, stack = None):
            _return = ""
            if type(stack) == str: stack = [stack]

            if   type(data) == list:
                if len(data) == 1:
                    _return += JSON2HTMLTableFilter(data[0], stack)
                else:
                    _return += '<ul class="via4">'
                    for item in data:
                        _return += ('<li>%s</li>'%JSON2HTMLTableFilter(item, stack))
                    _return += '</ul>'
            elif type(data) == dict:
                _return += '<table class="invisiTable">'
                for key, val in sorted(data.items()):
                    _return += '<tr><td><b>%s</b></td><td>%s</td></tr>'%(key, JSON2HTMLTableFilter(val, stack+[key])) 
                _return += '</table>'
            elif type(data) == str:
                if stack:
                    _return += "<a href='/link/"+doublequote('.'.join(stack))+"/"+doublequote(data)+"'>" #link opening
                    _return += "<span class='glyphicon glyphicon-link' aria-hidden='true'></span> </a>"
                _return += "<a target='_blank' href='%s'>%s</a>"%(data, data) if tk.isURL(data) else data
            _return += ""
            return _return
        return dict(JSON2HTMLTable=JSON2HTMLTableFilter)


    ##################
    # Error Messages #
    ##################
    def page_not_found(self, e):
        return render_template('404.html', minimal=self.minimal), 404

if __name__ == '__main__':
    server = Minimal()
    server.start()
