#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# Minimal web interface to cve-search to display the last entries
# and view a specific CVE.
#
# Software is free software released under the "Modified BSD license"
#

# Copyright (c) 2013-2016  Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2014-2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# imports
import jinja2
import os
import re
import subprocess
import sys
import urllib
_runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(_runPath, ".."))

from flask       import abort, jsonify, request, redirect, render_template, send_file, Response
from flask_login import LoginManager, current_user, login_user, logout_user, login_required
from io          import TextIOWrapper, BytesIO
from redis       import exceptions as redisExceptions

import sbin.db_blacklist as bl
import sbin.db_whitelist as wl

from lib.Authentication import AuthenticationHandler
from lib.Config         import Configuration
from lib.PluginManager  import PluginManager
from lib.User           import User
from web.minimal        import Minimal
from web.advanced_api   import Advanced_API

class Index(Minimal, Advanced_API):
    #############
    # Variables #
    #############

    def __init__(self):
        Advanced_API.__init__(self)
        Minimal.__init__(self)
        self.minimal = False
        self.auth_handler  = AuthenticationHandler()
        self.plugManager   = PluginManager()
        self.login_manager = LoginManager()
        self.plugManager.loadPlugins()
        self.login_manager.init_app(self.app)
        self.login_manager.user_loader(self.load_user)
        self.redisdb = Configuration.getRedisVendorConnection()

        self.defaultFilters.update({'blacklistSelect': 'on', 'whitelistSelect': 'on',
                                    'unlistedSelect': 'show',})
        self.args.update({'minimal': False})
        self.pluginArgs = {"current_user":   current_user, "plugin_manager": self.plugManager}


    #############
    # Functions #
    #############
    def generate_full_query(self, f):
        query = self.generate_minimal_query(f)
        if current_user.is_authenticated():
            if f['blacklistSelect'] == "on":
                regexes = self.db.Blacklist.rules(compiled=False)
                if len(regexes) != 0:
                    exp = "^(?!" + "|".join(regexes) + ")"
                    query.append({'$or': [{'vulnerable_configuration': re.compile(exp)},
                                          {'vulnerable_configuration': {'$exists': False}},
                                          {'vulnerable_configuration': []} ]})
            if f['whitelistSelect'] == "hide":
                regexes = self.db.Whitelist.rules(compiled=False)
                if len(regexes) != 0:
                    exp = "^(?!" + "|".join(regexes) + ")"
                    query.append({'$or': [{'vulnerable_configuration': re.compile(exp)},
                                          {'vulnerable_configuration': {'$exists': False}},
                                          {'vulnerable_configuration': []} ]})
            if f['unlistedSelect'] == "hide":
                wlregexes = self.db.Whitelist.rules()
                blregexes = self.db.Blacklist.rules()
                query.append({'$or': [{'vulnerable_configuration': {'$in': wlregexes}},
                                      {'vulnerable_configuration': {'$in': blregexes}}]})
        query.extend(self.plugManager.doFilter(f, **self.pluginArgs))
        return query


    def markCPEs(self, cve):
        blacklist = self.db.Blacklist.rules()
        whitelist = self.db.Whitelist.rules()

        for conf in cve.vulnerable_configuration:
            conf.list  = 'none'
            conf.match = 'none'
            for w in whitelist:
                if w.match(conf.id):
                    conf.list  = 'white'
                    conf.match = w
            for b in blacklist:
                if b.match(conf.id):
                    conf.list  = 'black'
                    conf.match = b
        return cve


    def filter_logic(self, filters, skip, limit=None):
        query = self.generate_full_query(filters)
        limit = limit if limit else self.args['pageLength']
        cve   = self.db.CVE.query(limit=limit, skip=skip, query=query)
        # marking relevant records
        if current_user.is_authenticated():
            if filters['whitelistSelect'] == "on":   cve = self.list_mark('white', cve)
            if filters['blacklistSelect'] == "mark": cve = self.list_mark('black', cve)
        self.plugManager.mark(cve, **self.pluginArgs)
        cve = list(cve)
        return cve


    def addCPEToList(self, cpe, listType, cpeType=None):
        if not cpeType: cpeType='cpe'

        if listType.lower() in ("blacklist", "black", "b", "bl"):
            return self.db.Blacklist.insert(cpe, cpeType)
        if listType.lower() in ("whitelist", "white", "w", "wl"):
            return self.db.Whitelist.insert(cpe, cpeType)


    def list_mark(self, listed, cveList):
        if listed not in ['white', 'black']: return cveList
        _list = self.db.Whitelist if listed == "white" else self.db.Blacklist
        items = _list.rules()
        # check the cpes (full or partially) in the black/whitelist
        for i, cve in enumerate(cveList):
            for c in cve.vulnerable_configuration:
                if any(regex.match(c.id) for regex in items):
                    setattr(cveList[i], listed+'listed', 'yes')
        return cveList


    def filterUpdateField(self, data):
        if not data: return data
        returnvalue = []
        for line in data.split("\n"):
            if (not line.startswith("[+]Success to create index") and
                not line == "Not modified" and
                not line.startswith("Starting")):
                    returnvalue.append(line)
        return "\n".join(returnvalue)


    def adminInfo(self, output=None):
        return {'stats':        self.db.db_info(True),
                'plugins':      self.plugManager.getPlugins(),
                'updateOutput': self.filterUpdateField(output),
                'token':        self.db.Users.getToken(current_user.id)}


    # user management
    def load_user(self, id):
        return User.get(id, self.auth_handler)


    ##########
    # ROUTES #
    ##########
    def index(self):
        cve = self.filter_logic(self.defaultFilters, 0)
        filters = self.plugManager.getFilters(**self.pluginArgs)
        return render_template('index.html', cve=cve, r=0, filters=filters, **self.args)

    # /
    def index_post(self):
        args = dict(self.getFilterSettingsFromPost(0), **self.args)
        filters = self.plugManager.getFilters(**self.pluginArgs)
        return render_template('index.html', r=0, filters=filters, **args)

    # /r/<r>
    def index_filter_get(self, r):
        if not r or r < 0: r = 0
        cve = self.filter_logic(self.defaultFilters, r)
        filters = self.plugManager.getFilters(**self.pluginArgs)
        return render_template('index.html', cve=cve, r=r, filters=filters, **self.args)

    # /r/<r>
    def index_filter_post(self, r):
        if not r or r < 0: r = 0
        args = dict(self.getFilterSettingsFromPost(r), **self.args)
        filters = self.plugManager.getFilters(**self.pluginArgs)
        return render_template('index.html', r=r, filters=filters, **args)

    # /cve/<cveid>
    def cve(self, cveid):
        cve = self.api_cve(cveid)
        if cve is None:
            return render_template('error.html',status={'except':'cve-not-found','info':{'cve':cveid}})
        cve = self.markCPEs(cve)

        self.plugManager.onCVEOpen(cveid, **self.pluginArgs)
        pluginData = self.plugManager.cvePluginInfo(cveid, **self.pluginArgs)
        return render_template('cve.html', cve=cve, plugins=pluginData)


    # /_get_plugins
    def _get_plugins(self):
        if not current_user.is_authenticated(): # Don't show plugins requiring auth if not authenticated
            plugins = [{"name": x.getName(), "link": x.getUID()} for x in
                       self.plugManager.getWebPluginsWithPage(**self.pluginArgs) if not x.requiresAuth]
        else:
            plugins = [{"name": x.getName(), "link": x.getUID()} for x in
                       self.plugManager.getWebPluginsWithPage(**self.pluginArgs)]
        return jsonify({"plugins": plugins})


    # /plugin/_get_cve_actions
    def _get_cve_actions(self):
        cve = request.args.get('cve', type=str)
        if not current_user.is_authenticated(): # Don't show actions requiring auth if not authenticated
            actions = [x for x in self.plugManager.getCVEActions(cve, **self.pluginArgs) if not x['auth']]
        else:
            actions = self.plugManager.getCVEActions(cve, **self.pluginArgs)
        return jsonify({"actions": actions})


    # /plugin/<plugin>
    def openPlugin(self, plugin):
        if self.plugManager.requiresAuth(plugin) and not current_user.is_authenticated():
            return render_template("requiresAuth.html")
        else:
            data = self.plugManager.openPage(plugin, **self.pluginArgs)
            if data:
                page, data, mimetype = data
                if page:
                    try:
                        return render_template(page, **data)
                    except jinja2.exceptions.TemplateSyntaxError: return render_template("error.html", status={'except': 'plugin-page-corrupt'})
                    except jinja2.exceptions.TemplateNotFound:    return render_template("error.html", status={'except': 'plugin-page-not-found', 'page': page})
                elif data:
                    return Response(data, mimetype=mimetype)
            abort(404)


    # /plugin/<plugin>/subpage/<page>
    def openPluginSubpage(self, plugin, page):
        if self.plugManager.requiresAuth(plugin) and not current_user.is_authenticated():
            return render_template("requiresAuth.html")
        else:
            data = self.plugManager.openSubpage(plugin, page, **self.pluginArgs)
            if data:
                page, data, mimetype = data
                if page:
                    try:
                        return render_template(page, **data)
                    except jinja2.exceptions.TemplateSyntaxError: return render_template("error.html", status={'except': 'plugin-page-corrupt'})
                    except jinja2.exceptions.TemplateNotFound:    return render_template("error.html", status={'except': 'plugin-page-not-found', 'page': page})
                elif data:
                    return Response(data, mimetype=mimetype)
            abort(404)


    # /plugin/<plugin>/_cve_action/<action>
    def _jsonCVEAction(self, plugin, action):
        cve = request.args.get('cve', type=str)
        response = self.plugManager.onCVEAction(cve, plugin, action, fields=dict(request.args), **self.pluginArgs)
        if   type(response) is bool and response is True:
            return jsonify({'status': 'plugin_action_complete'})
        elif type(response) is bool and response is False or response is None:
            return jsonify({'status': 'plugin_action_failed'})
        elif type(response) is dict:
            return jsonify(response)


    # /admin
    # /admin/
    def admin(self):
        if Configuration.loginRequired():
            if not current_user.is_authenticated():
                return render_template('login.html')
        else:
            person = User.get("_dummy_", self.auth_handler)
            login_user(person)
        output = None
        if os.path.isfile(Configuration.getUpdateLogFile()):
            with open(Configuration.getUpdateLogFile()) as updateFile:
                separator="==========================\n"
                output=updateFile.read().split(separator)[-2:]
                output=separator+separator.join(output)
        return render_template('admin.html', status="default", **self.adminInfo(output))


    # /admin/change_pass
    @login_required
    def change_pass(self):
        current_pass = request.args.get('current_pass')
        new_pass     = request.args.get('new_pass')
        if current_user.authenticate(current_pass):
            if new_pass:
              self.db.Users.changePassword(current_user.id , new_pass)
              return jsonify({"status": "password_changed"})
            return jsonify({"status": "no_password"})
        else:
            return jsonify({"status": "wrong_user_pass"})

    # /admin/request_token
    @login_required
    def request_token(self):
        return jsonify({"token": self.db.Users.generateToken(current_user.id)})

    # /admin/updatedb
    @login_required
    def updatedb(self):
        process = subprocess.Popen([sys.executable, os.path.join(_runPath, "../sbin/db_updater.py"), "-civ"], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        out, err = process.communicate()
        output="%s\n\nErrors:\n%s"%(str(out,'utf-8'),str(err,'utf-8')) if err else str(out,'utf-8')
        return jsonify({"updateOutput": output, "status": "db_updated"})


    # /admin/whitelist
    # /admin/blacklist
    @login_required
    def listView(self):
        if request.url_rule.rule.split('/')[2].lower() == 'whitelist':
            return render_template('list.html', rules=self.db.Whitelist.get(), listType="Whitelist")
        else:
            return render_template('list.html', rules=self.db.Blacklist.get(), listType="Blacklist")


    # /admin/whitelist/import
    # /admin/blacklist/import
    @login_required
    def listImport(self, force=None, path=None):
        _list = request.url_rule.split('/')[2]
        file = request.files['file']
        force = request.form.get('force')
        count = self.db.Whitelist.size() if _list.lower == 'whitelist' else self.db.Blacklist.size()
        if (count == 0) | (not count) | (force == "f"):
            if _list.lower == 'whitelist':
                self.db.Whitelist.clear()
                wl.importWhitelist(TextIOWrapper(file.stream))
            else:
                self.db.Blacklist.clear()
                bl.importBlacklist(TextIOWrapper(file.stream))
            status = _list[0]+"l_imported"
        else:
            status = _list[0]+"l_already_filled"
        return render_template('admin.html', status=status, **self.adminInfo())


    # /admin/whitelist/export
    # /admin/blacklist/export
    @login_required
    def listExport(self, force=None, path=None):
        _list = request.url_rule.rule.split('/')[2]
        bytIO = BytesIO()
        data = wl.exportWhitelist() if _list.lower == 'whitelist' else bl.exportBlacklist()
        bytIO.write(bytes(data, "utf-8"))
        bytIO.seek(0)
        return send_file(bytIO, as_attachment=True, attachment_filename=_list+".txt")


    # /admin/whitelist/drop
    # /admin/blacklist/drop
    @login_required
    def listDrop(self):
        _list = request.url_rule.split('/')[2].lower()
        if _list == 'whitelist':
            self.db.Whitelist.clear()
        else:
            self.db.Blacklist.clear()
        return jsonify({"status": _list[0]+"l_dropped"})


    # /admin/addToList
    @login_required
    def listAdd(self):
        cpe = request.args.get('cpe')
        cpeType = request.args.get('type')
        lst = request.args.get('list')
        if cpe and cpeType and lst:
            status = "added_to_list" if self.addCPEToList(cpe, lst, cpeType) else "already_exists_in_list"
            returnList = self.db.Whitelist.get() if lst=="whitelist" else self.db.Blacklist.get()
            return jsonify({"status":status, "rules":returnList, "listType":lst.title()})
        else: return jsonify({"status": "could_not_add_to_list"})


    # /admin/removeFromList
    @login_required
    def listRemove(self):
        cpe = request.args.get('cpe', type=str)
        cpe = urllib.parse.quote_plus(cpe).lower()
        cpe = cpe.replace("%3a", ":")
        cpe = cpe.replace("%2f", "/")
        lst = request.args.get('list', type=str)
        if cpe and lst:
            result=self.db.Whitelist.remove(cpe) if lst.lower()=="whitelist" else self.db.Blacklist.remove(cpe)
            status = "removed_from_list" if (result > 0) else "already_removed_from_list"
        else:
            status = "invalid_cpe"
        returnList = self.db.Whitelist.get() if lst=="whitelist" else self.db.Blacklist.get()
        return jsonify({"status":status, "rules":returnList, "listType":lst.title()})


    # /admin/editInList
    @login_required
    def listEdit(self):
        old = request.args.get('oldCPE')
        new = request.args.get('cpe')
        lst = request.args.get('list')
        CPEType = request.args.get('type')
        if old and new:
            result = (self.db.Whitelist.update(old, new, CPEType) if lst=="whitelist" else
                      self.db.Blacklist.update(old, new, CPEType))
            status = "cpelist_updated" if (result) else "cpelist_update_failed"
        else:
            status = "invalid_cpe"
        returnList = self.db.Whitelist.get() if lst=="whitelist" else self.db.Blacklist.get()
        return jsonify({"rules":returnList, "status":status, "listType":lst})


    # /admin/listmanagement/<vendor>/<product>
    # /admin/listmanagement/<vendor>
    # /admin/listmanagement
    @login_required
    def listManagement(self, vendor=None, product=None):
        try:
            if product is None:
                # no product selected yet, so same function as /browse can be used
                if vendor:
                    vendor = urllib.parse.quote_plus(vendor).lower()
                browseList = self.api_browse(vendor)
                vendor = browseList["vendor"]
                product = browseList["product"]
                version = None
            else:
                # product selected, product versions required
                product = urllib.parse.quote_plus(product).lower()
                version = sorted(list(self.redisdb.smembers("p:" + product)))
            return render_template('listmanagement.html', vendor=vendor, product=product, version=version)
        except redisExceptions.ConnectionError:
            return render_template('error.html',
                                   status={'except':'redis-connection',
                                           'info':{'host':Configuration.getRedisHost(),'port':Configuration.getRedisPort()}})


    # /admin/listmanagement/add
    @login_required
    def listManagementAdd(self):
        # retrieve the separate item parts
        item     = request.args.get('item', type=str)
        listType = request.args.get('list', type=str)

        pattern = re.compile('^[a-z:/0-9.~_%-]+$')

        if pattern.match(item):
            item = item.split(":")
            added = False
            if len(item) == 1:
                # only vendor, so a check on cpe type is needed
                if self.redisdb.sismember("t:/o", item[0]):
                    if self.addCPEToList("cpe:2.3:o:" + item[0], listType): added = True
                if self.redisdb.sismember("t:/a", item[0]):
                    if self.addCPEToList("cpe:2.3:a:" + item[0], listType): added = True
                if self.redisdb.sismember("t:/h", item[0]):
                    if self.addCPEToList("cpe:2.3:h:" + item[0], listType): added = True
            elif 4 > len(item) > 1:
                # cpe type can be found with a mongo regex query
                result = self.db.CPE.get_regex("%s:%s"%(item[0], item[1]))
                if len(result) != 0:
                    prefix = ((result[0]).id)[:10]
                    if len(item) == 2:
                        if self.addCPEToList(prefix + item[0] + ":" + item[1], listType):
                            added = True
                    if len(item) == 3:
                        if self.addCPEToList(prefix + item[0] + ":" + item[1] + ":" + item[2], listType):
                            added = True
            status = "added_to_list" if added else "could_not_add_to_list"
        else:
            status = "invalid_cpe"
        j={"status":status, "listType":listType}
        return jsonify(j)


    # /login
    def login_check(self):
        # validate username and password
        username = request.form.get('username')
        password = request.form.get('password')
        person = User.get(username, self.auth_handler)
        try:
            if person and person.authenticate(password):
                login_user(person)
                return render_template('admin.html', status="logged_in", **self.adminInfo())
            else:
                return render_template('login.html', status="wrong_user_pass")
        except Exception as e:
            print(e)
            return render_template('login.html', status="outdated_database")


    # /logout
    @login_required
    def logout(self):
        logout_user()
        return redirect("/")


if __name__ == '__main__':
    server = Index()
    server.start()
