import unittest

import datetime

from lib.DatabaseLayer import DatabaseLayer
from lib.Objects       import CAPEC, CWE, CPE, CVE, Impact, Access


class TestObjects(unittest.TestCase):
    # Unittest functions
    def setUp(self):
        self.db = DatabaseLayer(_db="cvedb_test")
        self.capec1 = CAPEC(id="10000", name="test_capec",
                       summary="no summary",
                       prerequisites="No prerequisites",
                       solutions="There's no solution", weaknesses=[])
        self.cwe1 = CWE(id="10000", name="test_cwe",
                        description="test cwe", status="testing",
                        weakness='Testing')
        self.cpe1 = CPE(id="cpe:/a:test:test1", title="Test CPE 1",
                        references=[])
        self.cpe2 = CPE(id="cpe:2.3:a:test:test2", title="Test CPE 2",
                        references=[])
    def tearDown(self):
        del self.db
        del self.capec1
        del self.cwe1
        del self.cpe1
        del self.cpe2


    # Successful
    def test_create_capec(self):
        CAPEC(id="10000", name="test_capec", summary="no summary",
              prerequisites="No prerequisites",
              solutions="There's no solution", weaknesses=[])

    def test_create_cwe(self):
        CWE(id="10000", name="test_cwe", description="test cwe",
            status="testing", weakness='Testing')

    def test_create_cpe(self):
        CPE(id="cpe:/a:test:test1", title="Test CPE 1", references=[])
        CPE(id="cpe:2.3:a:test:test2", title="Test CPE 2", references=())

    def test_create_cve(self):
        CVE(id="CVE-0001-0001", cvss=0.1, summary="Test Vulnerability",
            vulnerable_configuration=[self.cpe1, self.cpe2],
            published=datetime.datetime(2017,1,1),
            impact=Impact("None", "None", "None"),
            access=Access("Low", "None", "Local"), cwe=self.cwe1)

    # Failed
    def test_fail_create_capec(self):
        try:
            CAPEC(id="10000", name=1, summary="no summary",
                  prerequisites="No prerequisites",
                  solutions="There's no solution", weaknesses=[])
        except TypeError as e:
            assert "name is supposed to be of type str, not int" == str(e)

    def test_fail_create_cwe(self):
        try:
            CWE(id="10000", name=1, description="test cwe",
                status="testing", weakness='Testing')
            self.fail("Object should not be created")
        except TypeError as e:
            assert "name is supposed to be of type str, not int" == str(e)

    def test_fail_create_cpe(self):
        try:
            CPE(id=1, title="Test CPE 2", references=())
            self.fail("Object should not be created")
        except TypeError as e:
            assert "id is supposed to be of type str, not int" == str(e)

    def test_fail_create_impact(self):
        try:
            Impact("test", "test", "test")
        except ValueError as e:
            assert "incorrect values given" == str(e)

    def test_fail_create_access(self):
        try:
            Access("test", "test", "test")
        except ValueError as e:
            assert "incorrect values given" == str(e)

    def test_fail_create_cve(self):
        try:
            CVE(id="CVE-0001-0001", cvss=0.1,
                summary="Test Vulnerability",
                vulnerable_configuration=[self.cpe1, "cpe:/a:test:test2"],
                published=datetime.datetime(2017,1,1),
                impact=Impact("None", "None", "None"),
                access=Access("Low", "None", "Local"), cwe=self.cwe1)
        except TypeError as e:
           assert "not all the items in the list are of type CPE" == str(e)



class TestDatabase(unittest.TestCase):
    # Unittest functions
    def setUp(self):
        self.db = DatabaseLayer(_db="cvedb_test")
        self.capec1 = CAPEC(id="10000", name="test_capec",
                       summary="no summary",
                       prerequisites="No prerequisites",
                       solutions="There's no solution",
                       weaknesses=["10000"])
        self.cwe1 = CWE(id="10000", name="test_cwe",
                        description="test cwe", status="testing",
                        weakness='Testing')
        self.cpe1 = CPE(id="cpe:/a:test:test1", title="Test CPE 1",
                        references=[])
        self.cpe2 = CPE(id="cpe:2.3:a:test:test2", title="Test CPE 2",
                        references=[])
        self.cve1 = CVE(id="CVE-0001-0001", cvss=0.1,
                        summary="Test Vulnerability",
                        vulnerable_configuration=[self.cpe1, self.cpe2],
                        published=datetime.datetime(2017,1,1),
                        impact=Impact("None", "None", "None"),
                        access=Access("Low", "None", "Local"),
                        cwe=self.cwe1)

    def tearDown(self):
        del self.db
        del self.capec1
        del self.cwe1
        del self.cpe1
        del self.cpe2
        del self.cve1

    # Insert Objects & Check
    def test_insert_capec(self):
        self.db.CAPEC.upsert(self.capec1)
        result = self.db.CAPEC.get(self.capec1.id)
        assert self.capec1.id        == result.id
        assert self.capec1.summary   == result.summary
        assert self.capec1.solutions == result.solutions

    def test_insert_cwe(self):
        self.db.CWE.upsert(self.cwe1)
        result = self.db.CWE.get(self.cwe1.id)
        assert self.cwe1.id           == result.id
        assert self.cwe1.name         == result.name
        assert self.cwe1.description  == result.description

    def test_insert_cpe(self):
        self.db.CPE.upsert([self.cpe1, self.cpe2])
        result = self.db.CPE.get(self.cpe2.id)
        assert self.cpe2.id    == result.id
        assert self.cpe2.title == result.title


    def test_insert_cve(self):
        self.db.CVE.upsert(self.cve1)
        result = self.db.CVE.get(self.cve1.id)
        assert self.cve1.id      == result.id
        assert self.cve1.summary == result.summary
        assert self.cve1.cvss    == result.cvss

class TestDatabaseLayer(unittest.TestCase):
    # Unittest functions
    def setUp(self):
        self.db = DatabaseLayer(_db="cvedb_test")
        self.capec1 = CAPEC(id="10000", name="test_capec",
                       summary="no summary",
                       prerequisites="No prerequisites",
                       solutions="There's no solution",
                       weaknesses=["10000"])
        self.cwe1 = CWE(id="10000", name="test_cwe",
                        description="test cwe", status="testing",
                        weakness='Testing')
        self.cpe1 = CPE(id="cpe:/a:test:test1", title="Test CPE 1",
                        references=[])
        self.cpe2 = CPE(id="cpe:2.3:a:test:test2", title="Test CPE 2",
                        references=[])
        self.cve1 = CVE(id="CVE-0001-0001", cvss=0.1,
                        summary="Test Vulnerability",
                        vulnerable_configuration=[self.cpe1, self.cpe2],
                        published=datetime.datetime(2017,1,1),
                        impact=Impact("None", "None", "None"),
                        access=Access("Low", "None", "Local"),
                        cwe=self.cwe1)

        self.db.CAPEC.upsert(self.capec1)
        self.db.CWE.upsert(self.cwe1)
        self.db.CPE.upsert([self.cpe1, self.cpe2])
        self.db.CVE.upsert(self.cve1)

    def tearDown(self):
        del self.db
        del self.capec1
        del self.cwe1
        del self.cpe1
        del self.cpe2
        del self.cve1

    def test_reconstruction_cve(self):
        result = self.db.CVE.get(self.cve1.id)

        assert self.cve1.cwe.name == result.cwe.name
        assert self.capec1.name   == result.cwe.capec[0].name
        # Needed because we can't compare object IDs
        vc1 = [(x.id, x.title) for x in
                self.cve1.vulnerable_configuration]
        vc2 = [(x.id, x.title) for x in result.vulnerable_configuration]
        self.assertListEqual(vc1, vc2)
        assert self.cve1.impact.integrity == result.impact.integrity
        assert self.cve1.access.vector    == result.access.vector
