import unittest

import datetime
import json

from urllib.request import urlopen

from lib.DatabaseLayer import DatabaseLayer
from lib.Objects       import CAPEC, CWE, CPE, CVE, Impact, Access

class TestAPI(unittest.TestCase):
    # Unittest functions
    def setUp(self):
        self.db = DatabaseLayer(_db="cvedb_test")
        self.capec1 = CAPEC(id="10000", name="test_capec",
                       summary="no summary",
                       prerequisites="No prerequisites",
                       solutions="There's no solution",
                       weaknesses=["10000"])
        self.cwe1 = CWE(id="10000", name="test_cwe",
                        description="test cwe", status="testing",
                        weakness='Testing')
        self.cpe1 = CPE(id="cpe:/a:test:test1", title="Test CPE 1",
                        references=[])
        self.cpe2 = CPE(id="cpe:2.3:a:test:test2", title="Test CPE 2",
                        references=[])
        self.cve1 = CVE(id="CVE-0001-0001", cvss=0.1,
                        summary="Test Vulnerability",
                        vulnerable_configuration=[self.cpe1, self.cpe2],
                        published=datetime.datetime(2017,1,1),
                        impact=Impact("None", "None", "None"),
                        access=Access("Low", "None", "Local"),
                        cwe=self.cwe1)

        self.db.CAPEC.upsert(self.capec1)
        self.db.CWE.upsert(self.cwe1)
        self.db.CPE.upsert([self.cpe1, self.cpe2])
        self.db.CVE.upsert(self.cve1)

        self.exp_capec1 = self.capec1.dict()
        self.exp_cwe1 = self.cwe1.dict()
        self.exp_cve1 = self.cve1.dict()
        self.exp_cve1['Published'] = self.exp_cve1['Published'].isoformat()

    def tearDown(self):
        del self.db
        del self.capec1
        del self.cwe1
        del self.cpe1
        del self.cpe2
        del self.cve1

    # Helper Functions
    def response_check(self, url, expected):
        url = 'http://localhost:5000/api/%s'%url
        data = json.loads((urlopen(url).read()).decode('utf8'))
        self.assertEqual(expected, data)

    # Testing Functions
    def test_cpe2_3(self):
        expected = "cpe:2.3:a:test:test"
        self.response_check("cpe2.3/cpe:/a:test:test", expected)

    def test_cpe2_2(self):
        expected = "cpe:/a:test:test"
        self.response_check("cpe2.2/cpe:2.3:a:test:test", expected)

    def test_cvefor(self):
        expected = [self.exp_cve1]
        self.response_check("cvefor/%s"%self.cpe1.id, expected)

    def test_cve(self):
        expected = self.exp_cve1
        self.response_check("cve/%s"%self.cve1.id, expected)

    def test_cwe(self):
        expected = [self.exp_cwe1]
        self.response_check("cwe", expected)

    def test_cwe_number(self):
        expected = [self.exp_capec1]
        self.response_check("cwe/%s"%self.cwe1.id, expected)

    def test_capec(self):
        expected = self.exp_capec1
        self.response_check("capec/10000", expected)

    def test_last(self):
        expected = [self.exp_cve1]
        self.response_check("last", expected)
