#!/usr/bin/env python3
#
# Updater script of CVE/CPE database
#
# Copyright (c) 2012-2016  Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2014-2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# Imports
import argparse
import logging
import os
import shlex
import subprocess
import sys
import time

runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(runPath, ".."))

from lib.Config        import Configuration as conf
from lib.DatabaseLayer import DatabaseLayer
from lib.PluginManager import PluginManager


def _drop_collection(collection, logging=None):
    logging = logging or {}
    db = DatabaseLayer()
    cols = {'cve':      db.CVE.drop,             'cpe':   db.CPE.drop,
            'cpeother': db.CPE.alternative_drop, 'capec': db.CAPEC.drop,
            'cwe':      db.CWE.drop,             'via4':  db.VIA4.drop}

    if collection == "metadata":
        _log("Dropping metadata", **logging)
        db.drop_metadata()
    else:
        _log("Dropping collection: %s"%collection, **logging)
        if collection in cols.keys():
            cols[collection]()
        _log("%s dropped"%collection, **logging)


def _size_collection(collection):
    db = DatabaseLayer()
    cols = {'cve':      db.CVE.size,             'cpe':   db.CPE.size,
            'cpeother': db.CPE.alternative_size, 'capec': db.CAPEC.size,
            'cwe':      db.CWE.size,             'via4':  db.VIA4.size}
    return cols[collection]()


def _run_script(script, logging=None):
    logging = logging or {}
    _log('Starting %s'%script['name'], **logging)
    process = "%s %s"%(sys.executable, os.path.join(runPath, script['updater']))
    subprocess.Popen((shlex.split(process))).wait()
    if script['name'] != "redis-cache-cpe":
        _log("%s updated"%script['name'], **logging)


def _log(message="", toFile=False, verbose=False):
    if toFile:
        with open(Configuration.getUpdateLogFile(), "a") as log:
            log .write(message + "\n")
    if verbose:
        print (message)
    else:
        logging.info(message)


def update(minimal = False, logs     = False, drop = False,
           caching = False, indexing = False, plugins=False):
    # Create update list
    sources = [{'name': "cve",            'updater': "db_mgmt.py -u"},
               {'name': "cpe",            'updater': "db_mgmt_cpe_dictionary.py"},
               {'name': "cpeother",       'updater': "db_mgmt_cpe_other_dictionary.py"}]
    posts =   [{'name': 'ensureindex',    'updater': "db_mgmt_create_index.py"}]
    if not minimal:
        sources.extend([{'name': 'cwe',   'updater': "db_mgmt_cwe.py"},
                        {'name': 'capec', 'updater': "db_mgmt_capec.py"},
                        {'name': 'via4',  'updater': "db_mgmt_ref.py"} ])
    if caching:
        posts.insert(0, {'name': 'redis-cache-cpe',
                                          'updater': "db_cpe_browser.py"})
    # Make CVE populate if it gets dropped
    if drop:
        sources[0]['updater'] = "db_mgmt.py -p"

    # Start update
    if logging:
        _log("==========================", **logs)
        _log(time.strftime("%a %d %B %Y %H:%M", time.gmtime()), **logs)
        _log("==========================", **logs)

    # Drop if needed
    if drop:
        _drop_collection('metadata')

    newCVEs = 0
    # Sources
    for source in sources:
        # check if feed should be skipped
        if not conf.includesFeed(source['name']):
            continue
        if source['name'] is "cpeother":
            if _size_collection(source['name']) == 0:
                continue
        # start update
        # drop if needed
        if drop:
            _drop_collection(source['name'])
        _log('Starting %s'%source['name'], **logs)
        before = _size_collection(source['name'])
        _run_script(source)
        after  = _size_collection(source['name'])
        new = (after - before)
        if source['name'] == 'cve':
            newCVEs = new
        _log("%s has %s elements (%s update)"%(source['name'], after, new), **logs)
    # Posts
    for post in posts:
        _run_script(post)
    if indexing and newCVEs > 0:
        _run_script({'name': 'fulltext', 'updater': "db_fulltext.py -v -l%s"%newCVEs})

    if plugins:
        plugManager = PluginManager()
        plugManager.loadPlugins()
        plugins = plugManager.getPlugins()
        if len(plugins) != 0:
            for plug in plugins:
                _log("Starting " + plug.getName() + " plugin", **logs)
                message = plug.onDatabaseUpdate()
                if message: _log(message)


if __name__ == "__main__":
    argParser = argparse.ArgumentParser(description='Database updater for cve-search')
    argParser.add_argument('-v', action='store_true', help='Logging on stdout')
    argParser.add_argument('-l', action='store_true', help='Running at regular interval', default=False)
    argParser.add_argument('-i', action='store_true', help='Indexing new cves entries in the fulltext indexer', default=False)
    argParser.add_argument('-c', action='store_true', help='Enable CPE redis cache', default=False)
    argParser.add_argument('-f', action='store_true', help='Drop collections and force initial import', default=False)
    argParser.add_argument('-m', action='store_true', help='Minimal import', default=False)
    argParser.add_argument('-o', action='store_true', help='Save log output', default=False)
    argParser.add_argument('-p', action='store_true', help='Public sources only', default=False)
    args = argParser.parse_args()

    if args.f and args.l:
        print ("Drop collections and running in loop should not be used.")
        argParser.print_help()
        sys.exit(2)

    if not args.v:
        logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)

    logs = {'toFile':args.o, 'verbose': args.v }
    update(minimal=args.m, logs =logs,       drop   =args.f,
           caching=args.c, indexing=args.i,  plugins=(not args.p))
    while args.l:
        _log("", args.o, args.v)
        _log("Sleeping...", **logs)
        time.sleep(3600)
        update(minimal=args.m, logs =logs,       drop   =args.f,
               caching=args.c, indexing=args.i,  plugins=(not args.p))
