#!/usr/bin/env python3
#
# Import script of NIST CWE Common Weakness Enumeration.
#
# Until now, the import is only import Weakness description.
#
# The format is the following:
#
# { "_id" : ObjectId("52b70521b261026f36818515"), "weaknessabs" : "Variant",
# "name" : "ASP.NET Misconfiguration: Missing Custom Error Page",
# "description_summary" : "An ASP .NET application must enable custom error
# pages in order to prevent attackers from mining information from the
# framework's built-in responses.An ASP .NET application must enable custom
# error pages in order to prevent attackers from mining information from the
# framework's built-in responses.", "status" : "Draft", "id" : "12" }
#
# Software is free software released under the "Modified BSD license"
#
# Copyright (c) 2013-2014 	Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2015-2017		Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# Imports
import os
import sys
runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(runPath, ".."))

from dateutil.parser import parse as parse_datetime

from xml.sax import make_parser
from xml.sax.handler import ContentHandler
import argparse

from lib.Config        import Configuration
from lib.DatabaseLayer import DatabaseLayer
from lib.Objects       import CWE
from lib.ProgressBar   import progressbar

argparser = argparse.ArgumentParser(description='populate/update NIST CWE Common Weakness Enumeration database')
argparser.add_argument('-v', action='store_true', help='verbose output')
args = argparser.parse_args()

class CWEHandler(ContentHandler):
    def __init__(self):
        self.cwe = []
        self.description_summary_tag = False
        self.weakness_tag = False

    def startElement(self, name, attrs):
        if name == 'Weakness':
            self.weakness_tag = True
            self.cwe.append({'name':        attrs.get('Name'),
                             'id':          attrs.get('ID'),
                             'status':      attrs.get('Status'),
                             'weaknessabs': attrs.get('Weakness_Abstraction')})
        elif name == 'Description_Summary' and self.weakness_tag:
            self.description_summary_tag = True
            self.description_summary = ""

    def characters(self, ch):
        if self.description_summary_tag:
            self.description_summary += ch.replace("  ", " ")

    def endElement(self, name):
        if name == 'Description_Summary' and self.weakness_tag:
            self.description_summary_tag = False
            self.cwe[-1]['description_summary'] = self.description_summary.replace("\n", "")
        elif name == 'Weakness':
            self.weakness_tag = False

# make parser
parser = make_parser()
ch = CWEHandler()
parser.setContentHandler(ch)
db = DatabaseLayer()
# check modification date
try:
    (f, r) = Configuration.getFeedData('cwe')
except Exception as e:
    print(e)
    sys.exit("Cannot open url %s. Bad URL or not connected to the internet?"%(Configuration.getFeedURL("cwe")))
lastmodified = parse_datetime(r.headers['last-modified'], ignoretz=True)
i = db.CWE.updated()
if i is not None:
    if lastmodified == i:
        print("Not modified")
        sys.exit(0)


# parse xml and store in database
parser.parse(f)
cweList=[]
for cwe in progressbar(ch.cwe):
    data = CWE(cwe['id'], cwe['name'], cwe['description_summary'].replace("\t\t\t\t\t", " "),
               cwe['status'], cwe['weaknessabs'])
    if args.v:
        print (cwe)
    cweList.append(data)
db.CWE.upsert(cweList)

#update database info after successful program-run
db.CWE.updated(lastmodified)
