#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# Import script of cpe (Common Platform Enumeration) definition
# into a collection used for human readable lookup of product name.
# This is locating the cpe used inside the cve, but only the cpe
# not present inside the cpe official dictionary.
#
# Exemple:
#    CVE-2014-5446 -> cpe:/a:zohocorp:manageengine_netflow_analyzer:.*
#    but 'cpe:/a:zohocorp:manageengine_netflow_analyzer' is not in the
#    cpe official dictionary.
#
# Imported in cvedb in the collection named cpeother.
#
# The format of the collection is the following
#
# { "_id" : ObjectId("50a2739eae24ac2274eae7c0"),
#     "id" : "cpe:/a:zohocorp:manageengine_netflow_analyzer:10.2",
#      "title" : "cpe:/a:zohocorp:manageengine_netflow_analyzer:10.2"
# }
#
# Software is free software released under the "Modified BSD license"
#
# Copyright (c) 2014       psychedelys
# Copyright (c) 2014-2017  PidgeyL

# Imports
import os
import sys
runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(runPath, ".."))

from lib.DatabaseLayer import DatabaseLayer
from lib.Objects       import CPE
from lib.ProgressBar   import progressbar

db = DatabaseLayer()
# get dates
icve           = db.CVE.updated()
icpeo, indexed = db.CPE.alternative_updated()

if not indexed:
    indexed = 0

if icve and icpeo:
    if icpeo == icve:
        print("Not modified")
        sys.exit(0)

cves = db.CVE.query(skip=indexed, sort=("Published", "asc"))

if not cves:
    print ("Empty collections, import skipped")
    sys.exit(2)

unique = set()
for cve in progressbar(cves):
    for cpe in cve.vulnerable_configuration:
        unique.add(cpe.id)
indexed_cpe = set()
for cpe in unique:
    if db.CPE.get(cpe):
        unique.add(cpe)
for cpe in indexed_cpe:
    unique.remove(cpe)
if len(unique) > 0:
    db.CPE.alternative_upsert([CPE(x) for x in unique])

#update database info after successful program-run
db.CVE.updated(icve)
db.CPE.alternative_updated(icve, (indexed+len(cves)))
