#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# Fulltext indexer for the MongoDB CVE collection.
#
# The fulltext indexer is relying on Whoosh.
#
# The indexing is done by enumerating all items from
# the MongoDB CVE collection and indexing the summary text of each
# CVE. The Path of each document is the CVE-ID.
#
# Software is free software released under the "Modified BSD license"
#
# Copyright (c) 2012-2015  Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2015-2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# Imports
import argparse
import os
import sys
runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(runPath, ".."))

from whoosh.index  import create_in, exists_in, open_dir
from whoosh.fields import Schema, TEXT, ID

from lib.Config        import Configuration
from lib.DatabaseLayer import DatabaseLayer
from lib.ProgressBar   import progressbar

def get_schema_writer():
    indexpath = Configuration.getIndexdir()
    if not os.path.exists(indexpath):
        os.mkdir(indexpath)

    if not exists_in(indexpath):
        schema = Schema(title=TEXT(stored=True), path=ID(stored=True, unique=True), content=TEXT)
        ix = create_in(indexpath, schema)
    else:
        ix = open_dir(indexpath)
    try:
        return ix.writer()
    except:
        print ("Index is locked. Another db_fulltext process running?")
        sys.exit(1)

def index(limit=5, cpe_lookup=False, verbose=False):
    if limit == 0: limit = -1
    data = DatabaseLayer().CVE.last(limit=limit)
    for cve in progressbar(data,prefix="Processing"):
        writer = get_schema_writer()
        title  = cve.summary[:70]
        data   = cve.summary
        if cpe_lookup:
            for cpe in cve.vulnerable_configuration:
                data += " " + cpe.title
        if verbose:
            print ('Indexing CVE-ID ' + str(cve.id) + ' ' + title)
        writer.update_document(title=title, path=cve.id, content=data)
        writer.commit()

if __name__ == "__main__":
    argParser = argparse.ArgumentParser(description='Fulltext indexer for the MongoDB CVE collection')
    argParser.add_argument('-v', action='store_true', default=False, help='Verbose logging')
    argParser.add_argument('-l', default=5, type=int, help='Number of last entries to index (Default: 5) - 0 to index all documents')
    argParser.add_argument('-n', action='store_true', default=False, help='lookup complete cpe (Common Platform Enumeration) name for vulnerable configuration to add in the index')
    args = argParser.parse_args()

    index(limit=args.l, cpe_lookup=args.n, verbose=args.v)
