#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# CPEList class, used in black-and whitelists
#
# Software is free software released under the "Modified BSD license"
#

# Copyright (c) 2014-2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# Imports
# make sure these modules are available on your system
import json
import os
import sys
runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(runPath, ".."))

from lib.DatabaseLayer import DatabaseLayer

class CPEList:
    def __init__(self, collection, args):
        if   collection.lower() == "blacklist":
            self.collection = DatabaseLayer().Blacklist
        elif collection.lower() == "whitelist":
            self.collection = DatabaseLayer().Whitelist

        self.args = args
        self.db   = DatabaseLayer()

    # check if there are items in the collection
    def countItems(self):
        return self.collection.size()

    # check if a cpe is in the list
    def check(self, cpe):
        return self.collection.contains(cpe)

    # insert to database
    def insert(self, cpe, cpeType):
        self.collection.insert(cpe, cpeType)

    # remove a cpe from the list
    def remove(self, cpe):
        self.collection.remove(cpe)

    def update(self, cpeOld, cpeNew, cpeType):
        self.collection.update(cpeOld, cpeNew, cpeType)

    # drop the collection
    def dropCollection(self):
        self.collection.clear()

    # import a file that represents the cpe list
    def importList(self, importFile):
        count = 0
        # read each line from the import file and regex them to a cpe format
        try:
            for line in json.load(importFile):
                try:
                    t = line['type']
                    if t not in ['cpe', 'targetsoftware', 'targethardware']:
                        continue
                    cpe = line['id']
                    if 'comments' in line:
                        cpe += "#" + "#".join(line['comments'])
                    if self.insert(cpe, t):
                        count += 1
                except:
                    continue
            if self.args.v:
                print("%s products added to the list"%(count))
        except IOError:
            print('The list is corrupted!')
            sys.exit()

    # export a file that represents the cpe list
    def exportList(self, exportFile=None):
        listed = getattr(self.db, "get"+self.collection)()
        output = json.dumps(listed, sort_keys=True, indent=2)
        if exportFile == None:
            return output
        else:
            if not os.path.exists(exportFile) or self.args.f:
                export = open(exportFile, 'w')
                export.write(output)
                export.close()
                if self.args.v:
                    print("%s listed items exported"%(len(listed)))
            else:
                print("file already exists")

    # process the arguments and use it to take actions
    def process(self):
        if self.args.d:
            # drop the list
            self.dropCollection()
        elif self.args.i:
            # get import file
            textfile = self.args.i
            # check if the collection is empty
            count = self.countItems()
            if count > 0 and self.args.f is False:
                # not empty and not forced to drop
                print("list already populated")
            else:
                # drop collection and repopulate it
                self.dropCollection()
                self.importList(open(textfile))
        elif self.args.e:
            # get export file
            textfile = self.args.e
            self.exportList(textfile)
        elif self.args.a or self.args.A:
            # get list of cpe's to add
            if self.args.a:
                cpeList = self.args.a
            else:
                cpeList = [x for x in open(self.args.A[0])]
            # add each item from the list
            count = 0
            for cpeID in cpeList:
                if self.insert(cpeID,self.args.t):
                    count += 1
            if self.args.v:
                print("%s products added to the list"%(count))
        elif self.args.r or self.args.R:
            # get list of cpe's to remove
            if self.args.r:
                cpeList = self.args.r
            else:
                cpeList = [x for x in open(self.args.R[0])]
            # remove each item from the list
            count = 0
            for cpeID in cpeList:
                amount = self.remove(cpeID)
                count += amount
            if self.args.v:
                print("%s products removed from the list"%(count))
