#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# Output formats for CVE data
#
# Software is free software released under the "Modified BSD license"
#
# Copyright (c) 2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# Imports
import csv
import io
import json
import re

from bson                  import json_util
from xml.etree.ElementTree import Element, SubElement, tostring
from xml.sax.saxutils      import escape as SaxEscape

# Functions
def output(cves, output_type, **kwargs):
    if not isinstance(cves, list): cves = [cves]
    if   output_type == "csv":      return _csv(cves, **kwargs)
    elif output_type == "xml":      return _xml(cves, **kwargs)
    elif output_type == "json":     return _json(cves, **kwargs)
    elif output_type == "html":     return _html(cves, **kwargs)
    elif output_type == "cve-id":   return _cve_id(cves, **kwargs)
    elif output_type == "human":    return _human(cves, **kwargs)
    elif output_type == "asciidoc": return _asciidoc(cves, **kwargs)
    else: return "Unknown format"


def _csv(cves, namelookup=False, **kwargs):
    _out = io.StringIO()
    data = csv.writer(_out, delimiter='|', quotechar='|',quoting=csv.QUOTE_MINIMAL)
    for cve in cves:
        cpes = " ".join([x.title if namelookup else x.id for x
                         in cve.vulnerable_configuration])
        data.writerow((cve.id,      str(cve.published), str(cve.cvss),
                       cve.summary, cve.references, cpes))
    return _out.getvalue()


def _json(cves, indent=None, namelookup=False, **kwargs):
    data = []
    for cve in cves:
        c = cve.dict(human_dates=True)
        cpes = [c.title if namelookup else c.id
                for c in cve.vulnerable_configuration]
        c['vulnerable_configuration'] = cpes
        if kwargs.get('ranking') and hasattr(cve, 'ranking'):
            c['ranking']=list(cve.ranking)
        data.append(c)
    data = json.dumps(data, default=json_util.default, indent=indent)
    return data


def _xml(cves, namelookup=False, ranking=False, **kwargs):
    r = Element('cve-search')
    def _addElement(tag, val):
        c = SubElement(r, tag)
        c.text=val
    for cve in cves:
        _addElement('id',        cve.id)
        _addElement('Published', str(cve.published))
        _addElement('cvss',      str(cve.cvss))
        _addElement('summary',   SaxEscape(cve.summary))
        for ref in cve.references:
            _addElement('references', SaxEscape(ref))
        for cpe in cve.vulnerable_configuration:
            v = namelookup and cpe.title or cpe.id
            _addElement('vulnerable_configuration', SaxEscape(v))
        if ranking and "ranking" in hasattr(cve, 'ranking'):
            for rank in cve.ranking:
                for group in rank:
                    for organization, score in group.items():
                        _addElement(organization, str(score))
    return tostring(r).decode('utf-8')

def _html(cves, product=None, cveids=None, **kwargs):
    output = ""
    if product:
        output = "<html><body><h1>CVE search " + product + " </h1>"
    elif cveids:
        output = "<html><body><h1>CVE-ID " + str(cveids) + " </h1>"
    else:
        return output
    for cve in cves:
        output += "<h2>" + cve.id + "<br /></h2>CVSS score: "
        output += str(cve.cvss) + "<br />" + "<b>"
        output += str(cve.published) + "<b><br />" + cve.summary
        output += "<br /> References:<br />"
        for entry in cve.references:
            output += entry + "<br />"
        output += "<hr><hr></body></html>"
    return output


def _cve_id(cves, **kwargs):
    return "\n".join([cve.id for cve in cves])


def _human(cves, namelookup=True, ranking=False, **kwargs):
    output = ""
    for cve in cves:
        output += "CVE: \t%s\n"%cve.id
        output += "DATE:\t%s\n"%str(cve.published)
        output += "CVSS:\t%s\n"%str(cve.cvss)
        output += cve.summary+"\n"
        output += "\nReferences:\n"
        output +=   "-----------\n"
        for entry in cve.references:
            output += (entry + "\n")
        output += "\nVulnerable Configs:\n"
        output +=   "-------------------\n"
        for cpe in cve.vulnerable_configuration:
            if namelookup: output += cpe.title + "\n"
            else:          output += cpe.id + "\n"
        if ranking and "ranking" in hasattr(cve, 'ranking'):
            output += "\nRanking:\n"
            output +=   "--------\n"
            for rank in cve.ranking:
                for group in rank:
                    for organization, score in group.items():
                        output += organization + ": " + str(score)+ "\n"
        output += "\n\n"
    return output


def _asciidoc(cves, namelookup=True, api=None, **kwargs):
    output = ""
    api = api or "http://cve.circl.lu/"
    for cve in cves:
        output += "= Common Vulnerabilities and Exposures - %s\n"%cve.id
        output += "cve-search <%s/cve/%s>\n"%(api, cve.id)
        output += "%s,%s\n"%(cve.id, cve.modified.strftime("%Y-%m-%dT%H:%M:%S.%f"))
        output += ":toc:\n"
        output += "== %s Summary\n\n"%cve.id
        output += "%s\n\n"%cve.summary
        output += "== Vulnerable configurations\n\n"
        for cpe in cve.vulnerable_configuration:
            output += "* %s\n"% (namelookup and re.sub(r'\n', '-', cpe.title) or cpe.id)
        if cve.cvss:
            output += "\n== Common Vulnerability Scoring System\n"
            output += "CVSS value:: %s\n"%cve.cvss
        if cve.impact:
            output += "\n== Impact Metrics\n\n"
            output += "[cols=\"1,2\"]\n"
            output += "|===\n"
            for t in ["availability", "confidentiality", "integrity"]:
                output += "|%s\n"%t.title()
                output += "|%s\n"%getattr(cve.impact, t)
            output += "|===\n"
        if cve.access:
            output += "\n== Access to the vulnerability\n\n"
            output += "[cols=\"1,2\"]\n"
            output += "|===\n"
            for t in ["authentication", "complexity", "vector"]:
                output += "|%s\n"%t.title()
                output += "|%s\n"%getattr(cve.access, t)
            output += "|===\n"
        if cve.references:
            output += "\n== References\n"
            for ref in cve.references:
                output += "* %s\n"%ref
        output += "\n\n"
    return output.strip()
