#!/usr/bin/env python3
#
# search is the search component of cve-search querying the database
#
# Software is free software released under the "Modified BSD license"
#
# Copyright (c) 2012       Wim Remes
# Copyright (c) 2012-2015  Alexandre Dulaunoy - a@foo.be
# Copyright (c) 2015-2017  Pieter-Jan Moreels - pieterjan.moreels@gmail.com

# Imports
import argparse
import os
import re
import sys
runPath = os.path.dirname(os.path.realpath(__file__))
sys.path.append(os.path.join(runPath, ".."))

from lib.DatabaseLayer import DatabaseLayer
from lib.Query         import Query
from lib.Output        import output

if __name__ == '__main__':
    _ap = argparse.ArgumentParser(description='Search for vulnerabilities in the National Vulnerability DB. Data from http://nvd.nist.org.')
    _ap.add_argument('-p', type=str, help='S = search product, e.g. o:microsoft:windows_7 or o:cisco:ios:12.1')
    _ap.add_argument('-f', type=str, help='F = free text search in vulnerability summary')
    _ap.add_argument('-c', action='append', help='search one or more CVE-ID')
    _ap.add_argument('-o', type=str, default='human', help='O = output format [csv|html|json|xml|cveid|asciidoc]')
    _ap.add_argument('-l', action='store_true', help='sort in descending mode')
    _ap.add_argument('-n', action='store_true', help='lookup complete cpe (Common Platform Enumeration) name for vulnerable configuration')
    _ap.add_argument('-r', action='store_true', help='lookup ranking of vulnerable configuration')
    _ap.add_argument('-a', default=False, action='store_true', help='Lookup CAPEC for related CWE weaknesses')
    _ap.add_argument('-v', type=str, help='vendor name to lookup in reference URLs')
    _ap.add_argument('--api', type=str, help='When used, the script will query the specified API rather than the local instance')
    args = _ap.parse_args()

    db    = DatabaseLayer()
    query = Query(api=args.api)
    items = []
    kwargs  = {'namelookup': args.n,
               'ranking':    args.r,
               'capec':      args.a,
               'product':    args.p, # only used by html, otherwise ignored
               'cveids':     args.c} # only used by html, otherwise ignored

    # Fetch cves
    if args.p:
        items.extend(query.cveforcpe(args.p))
    if args.f:
        items.extend(query.search(args.f))
    if args.c:
        for cve in args.c:
            items.append(query.cve(cve))
    # Fetch extra info if needed
    for item in items:
        ranking = []
        for conf in item.vulnerable_configuration:
            if args.r: # Ranking lookup
                rank = db.CPE.ranking(cpeid=conf)
                if rank and rank not in ranking:
                    ranking.append(rank)
        if args.r: # Ranking lookup
            item['ranking'] = ranking

    # Print output
    print(output(items, args.o, **kwargs))
